<?php
// Include the centralized teacher layout
require_once __DIR__ . '/components/teacher_layout.php';

// Include ID encryption utility
require_once "../includes/id_encryption.php";

include_once "../includes/functions.php";
require_once "../auth/functions.php";

// Error handling and reporting during development
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Get course ID from URL - handle both encrypted and legacy IDs
$course_id = 0;
if (isset($_GET['id'])) {
    if (is_numeric($_GET['id'])) {
        // Legacy integer ID (for backward compatibility)
        $course_id = (int)$_GET['id'];
    } else {
        // Try to decrypt encrypted ID
        $course_id = IdEncryption::decrypt($_GET['id']);
        if ($course_id === false) {
            header("Location: courses.php?error=Invalid course ID");
            exit();
        }
    }
}

$teacher_id = $_SESSION['user_id'];

if ($course_id === 0) {
    header("Location: courses.php");
    exit();
}

// Fetch course details with validation
try {
    $sql = "SELECT c.*, cat.category_name 
            FROM courses c 
            LEFT JOIN categories cat ON c.category_id = cat.category_id 
            WHERE c.course_id = ? LIMIT 1";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $course_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $course = $result->fetch_assoc();
    
    if (!$course) {
        header("Location: courses.php?error=Course not found");
        exit();
    }
    
    // Get course statistics with error handling
    $stats = [
        'enrolled_students' => 0,
        'total_videos' => 0,
        'total_quizzes' => 0,
        'average_rating' => 0,
        'total_reviews' => 0
    ];
    
    // Check if database connection exists
    if (!$conn) {
        error_log("Database connection not available");
    } else {
        // Enrolled students count - try multiple table names
        $enrollment_tables = ['course_enrollments', 'enrollments', 'student_enrollments'];
        foreach ($enrollment_tables as $table) {
            $stmt = $conn->prepare("SELECT COUNT(*) as count FROM {$table} WHERE course_id = ?");
            if ($stmt) {
                $stmt->bind_param("i", $course_id);
                if ($stmt->execute()) {
                    $result = $stmt->get_result();
                    if ($result) {
                        $stats['enrolled_students'] = $result->fetch_assoc()['count'];
                        break;
                    }
                }
            }
        }
        
        // Videos count
        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM videos WHERE course_id = ?");
        if ($stmt) {
            $stmt->bind_param("i", $course_id);
            if ($stmt->execute()) {
                $result = $stmt->get_result();
                if ($result) {
                    $stats['total_videos'] = $result->fetch_assoc()['count'];
                }
            }
        }
        
        // Quizzes count
        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM quizzes WHERE course_id = ?");
        if ($stmt) {
            $stmt->bind_param("i", $course_id);
            if ($stmt->execute()) {
                $result = $stmt->get_result();
                if ($result) {
                    $stats['total_quizzes'] = $result->fetch_assoc()['count'];
                }
            }
        }
        
        // Average rating - try multiple table names
        $rating_tables = ['course_ratings', 'course_reviews', 'ratings', 'reviews'];
        foreach ($rating_tables as $table) {
            $stmt = $conn->prepare("SELECT AVG(rating) as avg_rating, COUNT(*) as total_reviews FROM {$table} WHERE course_id = ?");
            if ($stmt) {
                $stmt->bind_param("i", $course_id);
                if ($stmt->execute()) {
                    $result = $stmt->get_result();
                    if ($result) {
                        $rating_result = $result->fetch_assoc();
                        $stats['average_rating'] = $rating_result['avg_rating'] ?? 0;
                        $stats['total_reviews'] = $rating_result['total_reviews'] ?? 0;
                        break;
                    }
                }
            }
        }
    }
    
} catch(Exception $e) {
    error_log("Error fetching course data: " . $e->getMessage());
    header("Location: courses.php?error=Database error");
    exit();
}

// Set current page for sidebar
$current_page = 'view-course.php';
?>

<!-- FontAwesome is already loaded in teacher_layout.php -->
    
<style>
    /* Modern Course View - Panadite Design */
    * {
        font-family: 'Inter', 'Poppins', sans-serif !important;
    }

    :root {
        --primary-color: #4361ee;
        --secondary-color: #3f37c9;
        --success-color: #22c55e;
        --info-color: #06b6d4;
        --warning-color: #f59e0b;
        --danger-color: #ef4444;
        --light-color: #f8fafc;
        --dark-color: #1f2937;
    }

    body {
        background-color: var(--light-color) !important;
    }

    .container-fluid {
        padding: 0 30px !important;
    }
    
    /* Clean Course Header */
    .course-header {
        background: #ffffff !important;
        border: 1px solid #f1f5f9 !important;
        border-radius: 12px !important;
        padding: 24px !important;
        margin-bottom: 32px !important;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08) !important;
        position: relative !important;
    }
    
    .course-title {
        font-size: 24px !important;
        font-weight: 700 !important;
        color: #1f2937 !important;
        margin-bottom: 8px !important;
        line-height: 1.3 !important;
        display: flex !important;
        align-items: center !important;
    }

    .course-title i {
        color: var(--primary-color) !important;
        font-size: 28px !important;
        margin-right: 12px !important;
    }
    
    .course-description {
        font-size: 14px !important;
        color: #6b7280 !important;
        line-height: 1.6 !important;
        margin-bottom: 16px !important;
    }
    
    .course-meta {
        display: flex !important;
        gap: 16px !important;
        flex-wrap: wrap !important;
        margin-top: 0 !important;
    }
    
    .meta-item {
        display: flex;
        align-items: center;
        gap: 6px !important;
        background: #f8fafc !important;
        padding: 6px 12px !important;
        border-radius: 20px !important;
        font-size: 12px !important;
        font-weight: 500 !important;
        color: #6b7280 !important;
        border: 1px solid #f1f5f9 !important;
    }
    
    .meta-item i {
        font-size: 14px !important;
        color: var(--primary-color) !important;
    }
    
    /* Compact Stats Cards - 50% width container */
    .stats-container {
        width: 50% !important;
        margin-bottom: 32px !important;
    }
    
    .stats-row {
        display: flex !important;
        gap: 16px !important;
        flex-wrap: wrap !important;
    }
    
    .stat-card {
        background: #ffffff !important;
        border: 1px solid #f1f5f9 !important;
        border-radius: 12px !important;
        padding: 20px !important;
        flex: 1 !important;
        min-width: 140px !important;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08) !important;
        transition: all 0.2s ease !important;
        text-align: center !important;
    }

    .stat-card:hover {
        transform: translateY(-4px) !important;
        box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15) !important;
        border-color: #e5e7eb !important;
    }
    
    .stat-icon {
        width: 48px !important;
        height: 48px !important;
        border-radius: 10px !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        margin: 0 auto 12px !important;
        background: #f8fafc !important;
        border: 1px solid #f1f5f9 !important;
    }
    
    .stat-icon.students {
        background: rgba(34, 197, 94, 0.1) !important;
    }
    
    .stat-icon.videos {
        background: rgba(6, 182, 212, 0.1) !important;
    }
    
    .stat-icon.quizzes {
        background: rgba(245, 158, 11, 0.1) !important;
    }
    
    .stat-icon.rating {
        background: rgba(139, 92, 246, 0.1) !important;
    }
    
    .stat-icon i {
        font-size: 24px !important;
        line-height: 1 !important;
    }
    
    .stat-icon.students i {
        color: #22c55e !important;
    }
    
    .stat-icon.videos i {
        color: #06b6d4 !important;
    }
    
    .stat-icon.quizzes i {
        color: #f59e0b !important;
    }
    
    .stat-icon.rating i {
        color: #8b5cf6 !important;
    }
    
    .stat-number {
        font-size: 24px !important;
        font-weight: 700 !important;
        line-height: 1.2 !important;
        margin-bottom: 4px !important;
        color: #1f2937 !important;
    }
    
    .stat-label {
        font-size: 12px !important;
        font-weight: 500 !important;
        color: #6b7280 !important;
        text-transform: uppercase !important;
        letter-spacing: 0.5px !important;
        margin: 0 !important;
    }
    
    /* Modern Tab Navigation */
    .custom-tabs {
        background: #ffffff !important;
        border: 1px solid #f1f5f9 !important;
        border-radius: 12px !important;
        padding: 16px !important;
        margin-bottom: 24px !important;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08) !important;
    }
    
    .nav-tabs {
        border: none !important;
        padding: 0 !important;
        gap: 8px !important;
        justify-content: space-around !important;
    }
    
    .nav-tabs .nav-link {
        border: none !important;
        padding: 16px 30px !important;
        border-radius: 14px !important;
        color: var(--dark-color) !important;
        font-weight: 600 !important;
        transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        background: none !important;
        flex: 1 !important;
        text-align: center !important;
        margin: 0 !important;
        position: relative !important;
        z-index: 2 !important;
    }
    
    .nav-tabs .nav-link:hover {
        background: var(--light-color) !important;
        color: var(--primary-color) !important;
    }
    
    .nav-tabs .nav-link.active {
        background: #ffffff !important;
        color: var(--primary-color) !important;
        border: none !important;
        border-bottom: 2px solid var(--primary-color) !important;
        font-weight: 600 !important;
    }
    
    /* Modern Content Cards */
    .content-card {
        background: #ffffff !important;
        border: 1px solid #f1f5f9 !important;
        border-radius: 12px !important;
        padding: 24px !important;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08) !important;
        margin-bottom: 24px !important;
    }
    
    .content-card h4 {
        color: #1f2937 !important;
        font-weight: 600 !important;
        margin-bottom: 20px !important;
        font-size: 18px !important;
    }
    
    /* Modern Tables */
    .table-responsive {
        background: #ffffff !important;
        border-radius: 8px !important;
        border: 1px solid #f1f5f9 !important;
        overflow: hidden !important;
    }
    
    .table {
        margin: 0 !important;
        background: transparent !important;
    }
    
    .table thead th {
        background: #f8fafc !important;
        color: #374151 !important;
        font-weight: 600 !important;
        font-size: 13px !important;
        text-transform: uppercase !important;
        letter-spacing: 0.5px !important;
        border-bottom: 1px solid #f1f5f9 !important;
        padding: 16px !important;
    }
    
    .table tbody td {
        padding: 12px 16px !important;
        border-bottom: 1px solid #f9fafb !important;
        vertical-align: middle !important;
        font-size: 14px !important;
        color: #374151 !important;
    }

    .table tbody tr:hover {
        background: #f8faff !important;
    }

    .table tbody tr:last-child td {
        border-bottom: none;
    }
</style>

<!-- Course Header -->
<div class="course-header">
    <div class="d-flex justify-content-between align-items-start mb-3">
        <div class="flex-grow-1">
            <h1 class="course-title">
                <i class='bx bx-book-open'></i>
                <?= htmlspecialchars($course['title']) ?>
            </h1>
            <?php if (!empty($course['description'])): ?>
                <p class="course-description"><?= htmlspecialchars(substr($course['description'], 0, 150)) ?>...</p>
            <?php endif; ?>
        </div>
    </div>
    
    <div class="course-meta">
        <div class="meta-item">
            <i class='bx bx-user'></i>
            <span><?= htmlspecialchars($course['teacher_name'] ?? 'Teacher') ?></span>
        </div>
        <div class="meta-item">
            <i class='bx bx-dollar-circle'></i>
            <span>R<?= number_format($course['price'], 2) ?></span>
        </div>
        <div class="meta-item">
            <i class='bx bx-calendar'></i>
            <span><?= date('M j, Y', strtotime($course['created_at'])) ?></span>
        </div>
        <div class="meta-item">
            <i class='bx bx-check-circle'></i>
            <span><?= ucfirst($course['status']) ?></span>
        </div>
    </div>
</div>

<!-- Statistics Cards -->
<!-- Compact Statistics Cards - 50% Width -->
<div class="stats-container">
    <div class="stats-row">
        <div class="stat-card">
            <div class="stat-icon students">
                <i class='bx bx-user'></i>
            </div>
            <div class="stat-number"><?= $stats['enrolled_students'] ?></div>
            <p class="stat-label">Enrolled Students</p>
        </div>
        <div class="stat-card">
            <div class="stat-icon videos">
                <i class='bx bx-video'></i>
            </div>
            <div class="stat-number"><?= $stats['total_videos'] ?></div>
            <p class="stat-label">Videos</p>
        </div>
        <div class="stat-card">
            <div class="stat-icon quizzes">
                <i class='bx bx-list-check'></i>
            </div>
            <div class="stat-number"><?= $stats['total_quizzes'] ?></div>
            <p class="stat-label">Quizzes</p>
        </div>
        <div class="stat-card">
            <div class="stat-icon rating">
                <i class='bx bx-star'></i>
            </div>
            <div class="stat-number"><?= number_format($stats['average_rating'], 1) ?></div>
            <p class="stat-label">Average Rating</p>
        </div>
    </div>
</div>

<!-- Tabbed Content -->
<div class="custom-tabs">
    <ul class="nav nav-tabs" id="courseTabs" role="tablist">
        <li class="nav-item" role="presentation">
            <button class="nav-link active" id="videos-tab" data-bs-toggle="tab" 
                    data-bs-target="#videos" type="button" role="tab">
                <i class='bx bx-play-circle'></i> Videos (<?= $stats['total_videos'] ?>)
            </button>
        </li>
        <li class="nav-item" role="presentation">
            <button class="nav-link" id="quizzes-tab" data-bs-toggle="tab" 
                    data-bs-target="#quizzes" type="button" role="tab">
                <i class='bx bx-clipboard-check'></i> Quizzes (<?= $stats['total_quizzes'] ?>)
            </button>
        </li>
        <li class="nav-item" role="presentation">
            <button class="nav-link" id="students-tab" data-bs-toggle="tab" 
                    data-bs-target="#students" type="button" role="tab">
                <i class='bx bx-users'></i> Students (<?= $stats['enrolled_students'] ?>)
            </button>
        </li>
        <li class="nav-item" role="presentation">
            <button class="nav-link" id="reviews-tab" data-bs-toggle="tab" 
                    data-bs-target="#reviews" type="button" role="tab">
                <i class='bx bx-star'></i> Reviews (<?= $stats['total_reviews'] ?>)
            </button>
        </li>
    </ul>
    
    <div class="tab-content" id="courseTabContent">
        <!-- Videos Tab -->
        <div class="tab-pane fade show active" id="videos" role="tabpanel">
            <?php
            // Fetch videos directly - using simpler query first to debug
            try {
                // First check if videos table exists and has data
                $videos_stmt = $conn->prepare("SELECT video_id, title, description, duration, video_url, thumbnail_url, upload_date, views, is_published FROM videos WHERE course_id = ? ORDER BY video_id DESC");
                if (!$videos_stmt) {
                    // Try different field names
                    $videos_stmt = $conn->prepare("SELECT * FROM videos WHERE course_id = ?");
                }
                
                if ($videos_stmt) {
                    $videos_stmt->bind_param("i", $course_id);
                    if ($videos_stmt->execute()) {
                        $videos_result = $videos_stmt->get_result();
                        $videos = $videos_result->fetch_all(MYSQLI_ASSOC);
                        
                        if (!empty($videos)): ?>
                            <div class="table-container">
                                <table class="table">
                                    <thead>
                                        <tr>
                                            <th>Title</th>
                                            <th>Duration</th>
                                            <th>Views</th>
                                            <th>Status</th>
                                            <th>Upload Date</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($videos as $video): ?>
                                        <tr>
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <i class='bx bx-video me-2 text-primary'></i>
                                                    <?= htmlspecialchars($video['title']) ?>
                                                </div>
                                            </td>
                                            <td><?= htmlspecialchars($video['duration'] ?? 'N/A') ?></td>
                                            <td><?= number_format($video['views'] ?? 0) ?></td>
                                            <td>
                                                <span class="badge-custom <?= $video['is_published'] ? 'badge-success' : 'badge-warning' ?>">
                                                    <?= $video['is_published'] ? 'Published' : 'Draft' ?>
                                                </span>
                                            </td>
                                            <td><?= date('M j, Y', strtotime($video['upload_date'])) ?></td>
                                            <td>
                                                <div class="dropdown">
                                                    <button class="btn btn-sm btn-outline-secondary dropdown-toggle" data-bs-toggle="dropdown">
                                                        Actions
                                                    </button>
                                                    <ul class="dropdown-menu">
                                                        <li><a class="dropdown-item" href="<?php echo IdEncryption::generateUrl('edit-video.php', $video['video_id'], 'id'); ?>">
                                                            <i class='bx bx-edit'></i> Edit
                                                        </a></li>
                                                        <li><a class="dropdown-item" href="<?php echo IdEncryption::generateUrl('view-video.php', $video['video_id'], 'id'); ?>">
                                                            <i class='bx bx-play'></i> View
                                                        </a></li>
                                                    </ul>
                                                </div>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php else: ?>
                            <div class="empty-state">
                                <i class='bx bx-video'></i>
                                <h4>No Videos Yet</h4>
                                <p>Start building your course by adding video lessons.</p>
                                <a href="<?php echo IdEncryption::generateUrl('add-video.php', $course_id, 'course_id'); ?>" class="action-btn btn-primary-custom">
                                    <i class='bx bx-plus'></i> Add First Video
                                </a>
                            </div>
                        <?php endif;
                    }
                }
            } catch (Exception $e) {
                echo '<div class="alert alert-danger">Error loading videos</div>';
            }
            ?>
        </div>

        <!-- Quizzes Tab -->
        <div class="tab-pane fade" id="quizzes" role="tabpanel">
            <?php
            // Fetch quizzes directly
            try {
                $quizzes_stmt = $conn->prepare("SELECT * FROM quizzes WHERE course_id = ? ORDER BY created_at DESC");
                if ($quizzes_stmt) {
                    $quizzes_stmt->bind_param("i", $course_id);
                    if ($quizzes_stmt->execute()) {
                        $quizzes_result = $quizzes_stmt->get_result();
                        $quizzes = $quizzes_result->fetch_all(MYSQLI_ASSOC);
                        
                        if (!empty($quizzes)): ?>
                            <div class="table-container">
                                <table class="table">
                                    <thead>
                                        <tr>
                                            <th>Title</th>
                                            <th>Questions</th>
                                            <th>Time Limit</th>
                                            <th>Passing Score</th>
                                            <th>Status</th>
                                            <th>Created</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($quizzes as $quiz): ?>
                                        <tr>
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <i class='bx bx-list-check me-2 text-info'></i>
                                                    <?= htmlspecialchars($quiz['title']) ?>
                                                </div>
                                            </td>
                                            <td>
                                                <?php
                                                // Get question count
                                                $question_stmt = $conn->prepare("SELECT COUNT(*) as count FROM quiz_questions WHERE quiz_id = ?");
                                                if ($question_stmt) {
                                                    $question_stmt->bind_param("i", $quiz['quiz_id']);
                                                    $question_stmt->execute();
                                                    $question_count = $question_stmt->get_result()->fetch_assoc()['count'];
                                                    echo $question_count;
                                                } else {
                                                    echo 'N/A';
                                                }
                                                ?>
                                            </td>
                                            <td><?= $quiz['time_limit'] ? $quiz['time_limit'] . ' min' : 'Unlimited' ?></td>
                                            <td><?= $quiz['passing_score'] ?>%</td>
                                            <td>
                                                <span class="badge-custom <?= $quiz['is_published'] ? 'badge-success' : 'badge-warning' ?>">
                                                    <?= $quiz['is_published'] ? 'Published' : 'Draft' ?>
                                                </span>
                                            </td>
                                            <td><?= date('M j, Y', strtotime($quiz['created_at'])) ?></td>
                                            <td>
                                                <div class="dropdown">
                                                    <button class="btn btn-sm btn-outline-secondary dropdown-toggle" data-bs-toggle="dropdown">
                                                        Actions
                                                    </button>
                                                    <ul class="dropdown-menu">
                                                        <li><a class="dropdown-item" href="<?php echo IdEncryption::generateUrl('edit-quiz.php', $quiz['quiz_id'], 'id'); ?>">
                                                            <i class='bx bx-edit'></i> Edit
                                                        </a></li>
                                                        <li><a class="dropdown-item" href="<?php echo IdEncryption::generateUrl('quiz-submissions.php', $quiz['quiz_id'], 'quiz_id'); ?>">
                                                            <i class='bx bx-list-ul'></i> View Submissions
                                                        </a></li>
                                                    </ul>
                                                </div>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php else: ?>
                            <div class="empty-state">
                                <i class='bx bx-list-check'></i>
                                <h4>No Quizzes Yet</h4>
                                <p>Create quizzes to test your students' knowledge.</p>
                            </div>
                        <?php endif;
                    }
                }
            } catch (Exception $e) {
                echo '<div class="alert alert-danger">Error loading quizzes</div>';
            }
            ?>
        </div>

        <!-- Students Tab -->
        <div class="tab-pane fade" id="students" role="tabpanel">
            <?php
            // Fetch students enrolled in this course - try multiple table names like in stats
            try {
                $students = [];
                $enrollment_tables = ['course_enrollments', 'enrollments', 'student_enrollments'];
                
                foreach ($enrollment_tables as $table) {
                    $students_stmt = $conn->prepare("
                        SELECT e.*, u.user_id, u.first_name, u.last_name, u.email, u.profile_picture
                        FROM {$table} e
                        JOIN users u ON e.user_id = u.user_id OR e.student_id = u.user_id
                        WHERE e.course_id = ?
                        ORDER BY e.enrollment_date DESC
                    ");
                    
                    if ($students_stmt) {
                        $students_stmt->bind_param("i", $course_id);
                        if ($students_stmt->execute()) {
                            $students_result = $students_stmt->get_result();
                            $students = $students_result->fetch_all(MYSQLI_ASSOC);
                            if (!empty($students)) break;
                        }
                    }
                }
                
                if (!empty($students)): ?>
                            <div class="table-container">
                                <table class="table">
                                    <thead>
                                        <tr>
                                            <th>Student</th>
                                            <th>Email</th>
                                            <th>Enrolled Date</th>
                                            <th>Progress</th>
                                            <th>Status</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($students as $student): 
                                            $progress = (int)($student['progress'] ?? 0);
                                            $progress_class = $progress >= 80 ? 'text-success' : ($progress >= 50 ? 'text-warning' : 'text-danger');
                                        ?>
                                        <tr>
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <img src="<?= !empty($student['profile_picture']) ? htmlspecialchars($student['profile_picture']) : '/assets/images/default-avatar.png' ?>" 
                                                         alt="Profile" class="rounded-circle me-2" style="width: 32px; height: 32px; object-fit: cover;">
                                                    <span><?= htmlspecialchars($student['first_name'] . ' ' . $student['last_name']) ?></span>
                                                </div>
                                            </td>
                                            <td><?= htmlspecialchars($student['email']) ?></td>
                                            <td><?= date('M j, Y', strtotime($student['enrollment_date'])) ?></td>
                                            <td>
                                                <div class="progress-container">
                                                    <div class="progress" style="height: 8px;">
                                                        <div class="progress-bar <?= $progress >= 80 ? 'bg-success' : ($progress >= 50 ? 'bg-warning' : 'bg-danger') ?>" 
                                                             style="width: <?= $progress ?>%"></div>
                                                    </div>
                                                    <small class="<?= $progress_class ?>"><?= $progress ?>%</small>
                                                </div>
                                            </td>
                                            <td>
                                                <?php if ($student['completion_date']): ?>
                                                    <span class="badge-custom badge-success">Completed</span>
                                                <?php elseif ($progress >= 50): ?>
                                                    <span class="badge-custom badge-warning">In Progress</span>
                                                <?php else: ?>
                                                    <span class="badge-custom badge-secondary">Just Started</span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <button class="btn btn-sm btn-outline-info">
                                                    <i class='bx bx-message'></i> Message
                                                </button>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php else: ?>
                            <div class="empty-state">
                                <i class='bx bx-group'></i>
                                <h4>No Students Enrolled</h4>
                                <p>Students will appear here once they enroll in your course.</p>
                            </div>
                        <?php endif;
            } catch (Exception $e) {
                echo '<div class="alert alert-danger">Error loading students: ' . $e->getMessage() . '</div>';
            }
            ?>
        </div>

        <!-- Reviews Tab -->
        <div class="tab-pane fade" id="reviews" role="tabpanel">
            <?php
            // Fetch reviews for this course - use exact same table names as stats
            try {
                $rating_tables = ['course_ratings', 'course_reviews', 'ratings', 'reviews'];
                $reviews = [];
                
                foreach ($rating_tables as $table) {
                    $reviews_stmt = $conn->prepare("
                        SELECT r.*, u.first_name, u.last_name, u.profile_picture
                        FROM {$table} r 
                        LEFT JOIN users u ON r.user_id = u.user_id OR r.student_id = u.user_id
                        WHERE r.course_id = ? 
                        ORDER BY r.created_at DESC 
                        LIMIT 20
                    ");
                    
                    if ($reviews_stmt) {
                        $reviews_stmt->bind_param("i", $course_id);
                        if ($reviews_stmt->execute()) {
                            $result = $reviews_stmt->get_result();
                            $reviews = $result->fetch_all(MYSQLI_ASSOC);
                            if (!empty($reviews)) break;
                        }
                    }
                }
                
                if (!empty($reviews)): ?>
                    <div class="content-card">
                        <?php foreach ($reviews as $review): ?>
                        <div class="review-item mb-4 p-3 border rounded">
                            <div class="d-flex justify-content-between align-items-start mb-2">
                                <div class="d-flex align-items-center">
                                    <img src="<?= !empty($review['profile_picture']) ? htmlspecialchars($review['profile_picture']) : '/assets/images/default-avatar.png' ?>" 
                                         alt="Profile" class="rounded-circle me-2" style="width: 40px; height: 40px; object-fit: cover;">
                                    <div>
                                        <strong><?= htmlspecialchars(($review['first_name'] ?? '') . ' ' . ($review['last_name'] ?? '')) ?></strong>
                                        <div class="text-warning">
                                            <?php for($i = 1; $i <= 5; $i++): ?>
                                                <i class='bx <?= $i <= ($review['rating'] ?? 0) ? 'bxs-star' : 'bx-star' ?>'></i>
                                            <?php endfor; ?>
                                            <span class="ms-1"><?= number_format($review['rating'] ?? 0, 1) ?></span>
                                        </div>
                                    </div>
                                </div>
                                <small class="text-muted">
                                    <?= isset($review['created_at']) ? date('M j, Y', strtotime($review['created_at'])) : 'N/A' ?>
                                </small>
                            </div>
                            <p class="mb-0"><?= htmlspecialchars($review['review'] ?? $review['comment'] ?? $review['review_text'] ?? $review['feedback'] ?? 'No comment provided') ?></p>
                        </div>
                        <?php endforeach; ?>
                    </div>
                <?php else: ?>
                    <div class="empty-state">
                        <i class='bx bx-star'></i>
                        <h4>No Reviews Yet</h4>
                        <p>Student reviews and ratings will appear here once your course receives feedback.</p>
                    </div>
                <?php endif;
            } catch (Exception $e) {
                echo '<div class="alert alert-danger">Error loading reviews: ' . $e->getMessage() . '</div>';
            }
            ?>
        </div>
    </div>
</div>

<script>
// Simple tab functionality - no AJAX needed since content is loaded directly
document.addEventListener('DOMContentLoaded', function() {
    console.log('Course view page loaded successfully');
    
    // Optional: Add any page-specific interactions here
    // For now, Bootstrap handles all tab switching automatically
});
</script>

<style>
/* Targeted fixes for view-course page only */
.page-content {
    padding: 20px 30px;
}

.container-fluid {
    padding-left: 15px;
    padding-right: 15px;
}

.row {
    margin-left: 0;
    margin-right: 0;
}
</style>

<?php
// Include the centralized teacher layout end
require_once __DIR__ . '/components/teacher_layout_end.php';
?>
