<?php
// Set current page for active menu highlighting
$current_page = 'assignments.php';

// Include the centralized teacher layout
require_once __DIR__ . '/components/teacher_layout.php';

// Teacher Feedback File Handling Function
function handleTeacherFeedbackFiles($files, $assignment_id, $student_id, $conn) {
    $upload_dir = __DIR__ . '/../uploads/teacher_feedback/';
    $teacher_files = [];
    
    // Create directory if it doesn't exist
    if (!file_exists($upload_dir)) {
        mkdir($upload_dir, 0755, true);
    }
    
    // Get existing files from database
    $existing_query = "SELECT teacher_feedback_files FROM assignment_submissions WHERE assignment_id = ? AND student_id = ?";
    $stmt = $conn->prepare($existing_query);
    $stmt->bind_param("ii", $assignment_id, $student_id);
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    
    if (!empty($result['teacher_feedback_files'])) {
        $teacher_files = json_decode($result['teacher_feedback_files'], true) ?: [];
    }
    
    // Process uploaded files
    for ($i = 0; $i < count($files['name']); $i++) {
        if ($files['error'][$i] === UPLOAD_ERR_OK) {
            $original_name = $files['name'][$i];
            $tmp_name = $files['tmp_name'][$i];
            $file_size = $files['size'][$i];
            
            // Validate file size (10MB max)
            if ($file_size > 10 * 1024 * 1024) {
                continue; // Skip large files
            }
            
            // Generate unique filename
            $extension = pathinfo($original_name, PATHINFO_EXTENSION);
            $safe_filename = 'teacher_feedback_' . $assignment_id . '_' . $student_id . '_' . time() . '_' . $i . '.' . $extension;
            $file_path = $upload_dir . $safe_filename;
            
            // Move uploaded file
            if (move_uploaded_file($tmp_name, $file_path)) {
                $teacher_files[] = [
                    'original_name' => $original_name,
                    'file_path' => $safe_filename,
                    'file_size' => $file_size,
                    'upload_date' => date('Y-m-d H:i:s'),
                    'file_type' => $extension
                ];
            }
        }
    }
    
    return json_encode($teacher_files);
}

// Get assignment ID from URL
$assignment_id = intval($_GET['id'] ?? 0);
$userId = $_SESSION['user_id'];

if ($assignment_id <= 0) {
    header('Location: assignments.php');
    exit;
}

// Get assignment details and submissions
try {
    echo "<!-- Debug: Starting database queries -->\n";
    
    // Get assignment information with actual average score calculation
    $assignment_query = "
        SELECT a.*, c.title as course_title, c.course_id,
               COUNT(DISTINCT s.student_id) as total_submissions,
               ROUND(AVG(s.grade), 1) as avg_score
        FROM assignments a 
        JOIN courses c ON a.course_id = c.course_id 
        LEFT JOIN assignment_submissions s ON a.assignment_id = s.assignment_id 
        WHERE a.assignment_id = ? AND c.teacher_id = ?
        GROUP BY a.assignment_id
    ";
    
    echo "<!-- Debug: Preparing assignment query -->\n";
    $stmt = $conn->prepare($assignment_query);
    if (!$stmt) {
        die("Assignment query prepare failed: " . $conn->error);
    }
    
    echo "<!-- Debug: Executing assignment query for ID: $assignment_id, User: $userId -->\n";
    $stmt->bind_param("ii", $assignment_id, $userId);
    $stmt->execute();
    $assignment_result = $stmt->get_result();
    $assignment = $assignment_result->fetch_assoc();
    
    echo "<!-- Debug: Assignment found: " . ($assignment ? "YES" : "NO") . " -->\n";
    
    if (!$assignment) {
        echo "<!-- Debug: Assignment not found, redirecting -->\n";
        die("Assignment not found or you don't have permission to view it. <a href='assignments.php'>← Back to Assignments</a>");
    }
    
    // Get all submissions for this assignment
    echo "<!-- Debug: Getting submissions for assignment $assignment_id -->\n";
    $submissions_query = "
        SELECT s.submission_id, s.assignment_id, s.student_id, 
               s.submission_text, s.file_url, s.submission_date,
               s.grade, s.feedback, s.teacher_feedback_files,
               u.first_name, u.last_name, u.email
        FROM assignment_submissions s
        JOIN users u ON s.student_id = u.user_id
        WHERE s.assignment_id = ?
        ORDER BY s.submission_date DESC
    ";
    
    $stmt = $conn->prepare($submissions_query);
    if (!$stmt) {
        die("Submissions query prepare failed: " . $conn->error);
    }
    $stmt->bind_param("i", $assignment_id);
    $stmt->execute();
    $submissions_result = $stmt->get_result();
    $submissions = [];
    while ($row = $submissions_result->fetch_assoc()) {
        $submissions[] = $row;
    }
    
    echo "<!-- Debug: Found " . count($submissions) . " submissions -->\n";
    
    // Parse assignment data
    $submission_types = json_decode($assignment['submission_type'] ?? '[]', true) ?: [];
    $allowed_formats = json_decode($assignment['allowed_formats'] ?? '[]', true) ?: [];
    
    // Fix double-encoded formats if needed
    if (!empty($allowed_formats) && is_array($allowed_formats) && is_string($allowed_formats[0])) {
        $allowed_formats = json_decode($allowed_formats[0], true) ?: [];
    }
    
} catch (Exception $e) {
    error_log("Error in view-submissions.php: " . $e->getMessage());
    header('Location: assignments.php');
    exit;
}

// Handle score update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'update_score') {
    // Debug: Log all POST data
    error_log("POST data received: " . print_r($_POST, true));
    
    $student_id = intval($_POST['student_id'] ?? 0);
    $score = floatval($_POST['score'] ?? 0);
    $feedback = trim($_POST['feedback'] ?? '');
    
    error_log("Processing grade update - Student ID: $student_id, Score: $score, Assignment ID: $assignment_id");
    
    // Handle teacher feedback file uploads
    $teacher_files_json = null;
    if ($student_id > 0 && isset($_FILES['teacher_feedback_files'])) {
        $teacher_files_json = handleTeacherFeedbackFiles($_FILES['teacher_feedback_files'], $assignment_id, $student_id, $conn);
    }
    
    if ($student_id > 0) {
        try {
            // Update query to include teacher feedback files if provided
            if ($teacher_files_json !== null) {
                $update_query = "UPDATE assignment_submissions SET grade = ?, feedback = ?, teacher_feedback_files = ? WHERE assignment_id = ? AND student_id = ?";
                $stmt = $conn->prepare($update_query);
                if (!$stmt) {
                    throw new Exception("Prepare failed: " . $conn->error);
                }
                $stmt->bind_param("dssii", $score, $feedback, $teacher_files_json, $assignment_id, $student_id);
            } else {
                $update_query = "UPDATE assignment_submissions SET grade = ?, feedback = ? WHERE assignment_id = ? AND student_id = ?";
                $stmt = $conn->prepare($update_query);
                if (!$stmt) {
                    throw new Exception("Prepare failed: " . $conn->error);
                }
                $stmt->bind_param("dsii", $score, $feedback, $assignment_id, $student_id);
            }
            
            if ($stmt->execute()) {
                $affected_rows = $stmt->affected_rows;
                error_log("Grade update executed. Affected rows: $affected_rows");
                
                if ($affected_rows > 0) {
                    $success_message = "Grade updated successfully!";
                    // Add a small delay to ensure the update is committed
                    sleep(1);
                    header("Location: view-submissions.php?id=" . $assignment_id . "&success=1");
                    exit;
                } else {
                    $error_message = "No rows were updated. Check if submission exists.";
                    error_log("No rows affected in grade update");
                }
            } else {
                throw new Exception("Execute failed: " . $stmt->error);
            }
        } catch (Exception $e) {
            $error_message = "Error updating grade: " . $e->getMessage();
            error_log("Grade update error: " . $e->getMessage());
        }
    } else {
        $error_message = "Invalid student ID: $student_id";
        error_log("Invalid student ID received: $student_id");
    }
}

$page_title = "View Submissions - " . $assignment['title'];
?>

<!-- Page Content Starts Here -->
<style>
        :root {
            --panadite-primary: #2563eb;
            --panadite-secondary: #1d4ed8;
            --panadite-light: #dbeafe;
            --panadite-accent: #f59e0b;
            --text-primary: #1f2937;
            --text-secondary: #6b7280;
            --border-color: #e5e7eb;
            --success-color: #10b981;
            --error-color: #ef4444;
            --bg-light: #f8fafc;
        }

        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }

        body {
            font-family: 'Inter', sans-serif;
            background: var(--bg-light);
            color: var(--text-primary);
            line-height: 1.6;
        }

        .header {
            background: linear-gradient(135deg, var(--panadite-primary), var(--panadite-secondary));
            color: white;
            padding: 2rem 0;
            text-align: center;
        }

        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 1rem;
        }

        .back-link {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            color: white;
            text-decoration: none;
            margin-bottom: 1rem;
            padding: 0.5rem 1rem;
            border-radius: 8px;
            background: rgba(255, 255, 255, 0.1);
            transition: background 0.3s;
        }

        .back-link:hover {
            background: rgba(255, 255, 255, 0.2);
            color: white;
            text-decoration: none;
        }

        .assignment-info {
            background: white;
            border-radius: 12px;
            padding: 2rem;
            margin: 2rem 0;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
        }

        .assignment-meta {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
            margin-top: 1rem;
        }

        .meta-item {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.75rem;
            background: var(--bg-light);
            border-radius: 8px;
        }

        .submissions-section {
            background: white;
            border-radius: 12px;
            padding: 2rem;
            margin: 2rem 0;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
        }

        .submissions-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1.5rem;
            padding-bottom: 1rem;
            border-bottom: 2px solid var(--border-color);
        }

        .submission-card {
            border: 1px solid var(--border-color);
            border-radius: 12px;
            margin-bottom: 1.5rem;
            overflow: hidden;
            transition: transform 0.2s, box-shadow 0.2s;
        }

        .submission-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.1);
        }

        .submission-header {
            background: var(--bg-light);
            padding: 1.5rem;
            border-bottom: 1px solid var(--border-color);
        }

        .student-info {
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .submission-content {
            padding: 1.5rem;
        }

        .files-list {
            margin: 1rem 0;
        }

        .file-item {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.75rem;
            background: var(--bg-light);
            border-radius: 8px;
            margin-bottom: 0.5rem;
        }

        .grading-section {
            background: #f8f9fa;
            padding: 1.5rem;
            border-top: 1px solid var(--border-color);
        }

        .grading-form {
            display: grid;
            grid-template-columns: 1fr 1fr auto;
            gap: 1rem;
            align-items: end;
        }

        .form-group {
            display: flex;
            flex-direction: column;
            gap: 0.5rem;
        }

        .form-control {
            padding: 0.75rem;
            border: 1px solid var(--border-color);
            border-radius: 8px;
            font-family: inherit;
        }

        .btn {
            padding: 0.75rem 1.5rem;
            border: none;
            border-radius: 8px;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.3s;
        }

        .btn-primary {
            background: var(--panadite-primary);
            color: white;
        }

        .btn-primary:hover {
            background: var(--panadite-secondary);
        }

        .badge {
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 500;
        }

        .badge-success {
            background: #d4edda;
            color: #155724;
        }

        .badge-warning {
            background: #fff3cd;
            color: #856404;
        }

        .badge-secondary {
            background: #e2e3e5;
            color: #383d41;
        }

        .no-submissions {
            text-align: center;
            padding: 3rem;
            color: var(--text-secondary);
        }

        .download-btn {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.5rem 1rem;
            background: var(--panadite-primary);
            color: white;
            text-decoration: none;
            border-radius: 6px;
            font-size: 0.9rem;
            transition: background 0.3s;
        }

        .download-btn:hover {
            background: var(--panadite-secondary);
            color: white;
            text-decoration: none;
        }

        @media (max-width: 768px) {
            .grading-form {
                grid-template-columns: 1fr;
            }
            
            .assignment-meta {
                grid-template-columns: 1fr;
            }
        }
    </style>

<!-- Main Content Container -->
<div class="container-fluid mt-4">
    <div class="row">
        <div class="col-12">
            <!-- Page Header -->
            <div class="d-flex justify-content-between align-items-center mb-4">
                <div>
                    <h2 class="mb-1">View Submissions</h2>
                    <nav aria-label="breadcrumb">
                        <ol class="breadcrumb">
                            <li class="breadcrumb-item"><a href="assignments.php">Assignments</a></li>
                            <li class="breadcrumb-item active" aria-current="page"><?= htmlspecialchars($assignment['title']) ?></li>
                        </ol>
                    </nav>
                </div>
                <a href="assignments.php" class="btn btn-outline-primary me-2">
                    <i class="bx bx-arrow-back"></i> Back to Assignments
                </a>
                <a href="grades.php?assignment_id=<?= $assignment_id ?>" class="btn btn-success me-2">
                    <i class="bx bx-list-ul"></i> View All Grades
                </a>
                <a href="grades.php" class="btn btn-outline-secondary">
                    <i class="bx bx-bar-chart-alt-2"></i> Grades Overview
                </a>
            </div>

    <div class="container">
        <!-- Success/Error Messages -->
        <?php if (isset($success_message)): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <i class="bx bx-check-circle"></i> <?= htmlspecialchars($success_message) ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <?php if (isset($error_message)): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <i class="bx bx-error-circle"></i> <?= htmlspecialchars($error_message) ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <?php if (isset($_GET['success']) && $_GET['success'] == '1'): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <i class="bx bx-check-circle"></i> Grade updated successfully!
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <!-- Assignment Information -->
        <div class="assignment-info">
            <h2><i class="bx bx-info-circle"></i> <?= htmlspecialchars($assignment['title']) ?> Assignment Details</h2>
            
            <div class="assignment-meta">
                <div class="meta-item">
                    <i class="bx bx-calendar" style="color: var(--panadite-primary);"></i>
                    <div>
                        <strong>Due Date:</strong><br>
                        <?= date('M d, Y g:i A', strtotime($assignment['due_date'])) ?>
                    </div>
                </div>
                
                <div class="meta-item">
                    <i class="bx bx-star" style="color: var(--panadite-accent);"></i>
                    <div>
                        <strong>Points:</strong><br>
                        <?= $assignment['points'] ?> points
                    </div>
                </div>
                
                <div class="meta-item">
                    <i class="bx bx-user-check" style="color: var(--success-color);"></i>
                    <div>
                        <strong>Submissions:</strong><br>
                        <?= $assignment['total_submissions'] ?> received
                    </div>
                </div>
                
                <div class="meta-item">
                    <i class="bx bx-trending-up" style="color: var(--panadite-primary);"></i>
                    <div>
                        <strong>Average Score:</strong><br>
                        <?= $assignment['avg_score'] ? $assignment['avg_score'] . '%' : 'Not graded' ?>
                    </div>
                </div>
            </div>

            <?php if (!empty($submission_types)): ?>
            <div style="margin-top: 1.5rem;">
                <strong>Required Submission Types:</strong>
                <div style="margin-top: 0.5rem;">
                    <?php foreach ($submission_types as $type): ?>
                        <span class="badge badge-secondary" style="margin-right: 0.5rem;">
                            <?= ucwords(str_replace('_', ' ', $type)) ?>
                        </span>
                    <?php endforeach; ?>
                </div>
            </div>
            <?php endif; ?>

            <?php if (!empty($allowed_formats)): ?>
            <div style="margin-top: 1rem;">
                <strong>Allowed File Formats:</strong>
                <span style="color: var(--text-secondary);"><?= strtoupper(implode(', ', $allowed_formats)) ?></span>
            </div>
            <?php endif; ?>
        </div>

        <!-- Submissions Section -->
        <div class="submissions-section">
            <div class="submissions-header">
                <h2><i class="bx bx-file-blank"></i> Student Submissions</h2>
                <?php if (count($submissions) > 0): ?>
                    <div>
                        <span class="badge badge-success"><?= count($submissions) ?> submissions</span>
                    </div>
                <?php endif; ?>
            </div>

            <?php if (empty($submissions)): ?>
                <div class="no-submissions">
                    <i class="bx bx-inbox" style="font-size: 4rem; color: var(--text-secondary); margin-bottom: 1rem;"></i>
                    <h3>No submissions yet</h3>
                    <p>Students haven't submitted any work for this assignment.</p>
                </div>
            <?php else: ?>
                <?php foreach ($submissions as $submission): ?>
                    <div class="submission-card">
                        <div class="submission-header">
                            <div class="student-info">
                                <div>
                                    <h4><?= htmlspecialchars($submission['first_name'] . ' ' . $submission['last_name']) ?></h4>
                                    <p style="color: var(--text-secondary); margin: 0;"><?= htmlspecialchars($submission['email']) ?></p>
                                </div>
                                <div style="text-align: right;">
                                    <div style="color: var(--text-secondary); font-size: 0.9rem;">
                                        Submitted: <?= date('M d, Y g:i A', strtotime($submission['submission_date'])) ?>
                                    </div>
                                    <?php if ($submission['grade'] !== null && $submission['grade'] != ''): ?>
                                        <div style="font-weight: 600; color: var(--panadite-primary); margin-top: 0.25rem;">
                                            Score: <?= $submission['grade'] ?>%
                                        </div>
                                    <?php else: ?>
                                        <span class="badge badge-warning">Not Graded</span>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>

                        <div class="submission-content">
                            <?php if (!empty($submission['submission_text'])): ?>
                                <div style="margin-bottom: 1.5rem;">
                                    <h5><i class="bx bx-text"></i> Text Submission:</h5>
                                    <div style="background: var(--bg-light); padding: 1rem; border-radius: 8px; margin-top: 0.5rem; white-space: pre-wrap;">
                                        <?= htmlspecialchars($submission['submission_text']) ?>
                                    </div>
                                </div>
                            <?php endif; ?>

                            <?php if (!empty($submission['file_url'])): ?>
                                <?php
                                $file_url_raw = $submission['file_url'];
                                $files = null;
                                $display_files = [];
                                
                                // Method 1: Try JSON format (new format)
                                $files = json_decode($file_url_raw, true);
                                
                                // Method 2: If failed, try unescaping JSON
                                if (!is_array($files)) {
                                    $files = json_decode(stripslashes($file_url_raw), true);
                                }
                                
                                // Method 3: Try fixing common JSON issues
                                if (!is_array($files)) {
                                    $cleaned = str_replace(['\\"', "\\'"], ['"', "'"], $file_url_raw);
                                    $files = json_decode($cleaned, true);
                                }
                                
                                // Method 4: Handle truncated JSON (fix incomplete data)
                                if (!is_array($files) && substr($file_url_raw, 0, 1) === '[') {
                                    $last_brace = strrpos($file_url_raw, '}');
                                    if ($last_brace !== false) {
                                        $fixed_json = substr($file_url_raw, 0, $last_brace + 1) . ']';
                                        $files = json_decode($fixed_json, true);
                                    }
                                }
                                
                                // Method 5: Handle LEGACY format (simple string filename)
                                if (!is_array($files)) {
                                    // Legacy format: simple filename string
                                    $filename = trim($file_url_raw);
                                    if (!empty($filename)) {
                                        // Extract original name from legacy filename
                                        $original_name = $filename;
                                        // Try to get a more readable name
                                        if (preg_match('/assignment_\d+_student_\d+_\d+\.(\w+)$/', $filename, $matches)) {
                                            $extension = $matches[1];
                                            $original_name = "Submitted File." . $extension;
                                        }
                                        
                                        $display_files[] = [
                                            'original_name' => $original_name,
                                            'file_path' => $filename,
                                            'type' => 'legacy'
                                        ];
                                    }
                                } else {
                                    // JSON format worked - use parsed files
                                    $display_files = $files;
                                }
                                
                                if (!empty($display_files)):
                                ?>
                                    <div class="files-list">
                                        <h6><i class="bx bx-file"></i> Uploaded Files:</h6>
                                        <?php foreach ($display_files as $file): ?>
                                            <div class="file-item mb-2">
                                                <i class="bx bx-file-blank me-2"></i>
                                                <span class="file-name"><?= htmlspecialchars($file['original_name'] ?? 'Unknown file') ?></span>
                                                <a href="../uploads/assignments/<?= htmlspecialchars($file['file_path']) ?>" 
                                                   class="btn btn-sm btn-outline-primary ms-2" 
                                                   download="<?= htmlspecialchars($file['original_name'] ?? 'file') ?>">
                                                    <i class="bx bx-download"></i> Download
                                                </a>
                                            </div>
                                        <?php endforeach; ?>
                                    </div>
                                <?php else: ?>
                                    <p class="text-muted">No files attached.</p>
                                <?php endif; ?>
                            <?php endif; ?>

                            <?php if (!empty($submission['feedback'])): ?>
                                <div style="margin-top: 1.5rem;">
                                    <h5><i class="bx bx-message-detail"></i> Instructor Feedback:</h5>
                                    <div style="background: #fff3cd; padding: 1rem; border-radius: 8px; margin-top: 0.5rem; border-left: 4px solid var(--panadite-accent);">
                                        <?= nl2br(htmlspecialchars($submission['feedback'])) ?>
                                    </div>
                                </div>
                            <?php endif; ?>
                        </div>

                        <!-- Grading Section -->
                        <div class="grading-section">
                            <h5><i class="bx bx-edit"></i> Grade Submission</h5>
                            <form method="POST" action="" enctype="multipart/form-data">
                                <input type="hidden" name="action" value="update_score">
                                <input type="hidden" name="student_id" value="<?= $submission['student_id'] ?>">
                                
                                <div class="form-group">
                                    <label for="score_<?= $submission['student_id'] ?>">Score (%):</label>
                                    <input type="number" 
                                           id="score_<?= $submission['student_id'] ?>" 
                                           name="score" 
                                           min="0" 
                                           max="100" 
                                           step="0.1"
                                           value="<?= $submission['grade'] ?? '' ?>" 
                                           class="form-control" 
                                           placeholder="Enter score">
                                </div>
                                
                                <div class="form-group">
                                    <label for="feedback_<?= $submission['student_id'] ?>">Feedback:</label>
                                    <textarea id="feedback_<?= $submission['student_id'] ?>" 
                                              name="feedback" 
                                              class="form-control" 
                                              rows="3" 
                                              placeholder="Enter feedback for student"><?= htmlspecialchars($submission['feedback'] ?? '') ?></textarea>
                                </div>

                                <!-- Teacher Feedback Files Section -->
                                <div class="form-group">
                                    <label class="form-label">
                                        <i class="bx bx-file-plus"></i> Teacher Feedback Files
                                        <small class="text-muted">(Optional - Add documents, images, videos for student)</small>
                                    </label>
                                    
                                    <!-- Current Files Display -->
                                    <?php 
                                    $teacher_files = [];
                                    if (!empty($submission['teacher_feedback_files'])) {
                                        $teacher_files = json_decode($submission['teacher_feedback_files'], true) ?: [];
                                    }
                                    ?>
                                    
                                    <?php if (!empty($teacher_files)): ?>
                                        <div class="current-teacher-files mb-3">
                                            <h6 class="text-success"><i class="bx bx-check-circle"></i> Current Teacher Files:</h6>
                                            <?php foreach ($teacher_files as $index => $file): ?>
                                                <div class="teacher-file-item mb-3 p-3 bg-light rounded border">
                                                    <div class="d-flex align-items-center mb-2">
                                                        <?php
                                                        $file_ext = strtolower($file['file_type'] ?? pathinfo($file['original_name'], PATHINFO_EXTENSION));
                                                        $file_icon = 'bx-file';
                                                        $file_color = '#6c757d';
                                                        
                                                        switch($file_ext) {
                                                            case 'pdf': $file_icon = 'bx-file-blank'; $file_color = '#dc3545'; break;
                                                            case 'doc': case 'docx': $file_icon = 'bx-file-doc'; $file_color = '#0d6efd'; break;
                                                            case 'xls': case 'xlsx': $file_icon = 'bx-spreadsheet'; $file_color = '#198754'; break;
                                                            case 'ppt': case 'pptx': $file_icon = 'bx-slideshow'; $file_color = '#fd7e14'; break;
                                                            case 'jpg': case 'jpeg': case 'png': case 'gif': $file_icon = 'bx-image'; $file_color = '#6f42c1'; break;
                                                            case 'mp4': case 'mov': case 'avi': $file_icon = 'bx-video'; $file_color = '#e91e63'; break;
                                                            case 'mp3': case 'wav': $file_icon = 'bx-music'; $file_color = '#9c27b0'; break;
                                                        }
                                                        ?>
                                                        <i class="bx <?= $file_icon ?> me-2" style="color: <?= $file_color ?>; font-size: 1.5rem;"></i>
                                                        <div class="flex-grow-1">
                                                            <strong><?= htmlspecialchars($file['original_name'] ?? 'Unknown file') ?></strong>
                                                            <br><small class="text-muted">
                                                                <?= isset($file['file_size']) ? number_format($file['file_size'] / 1024, 1) . ' KB' : '' ?>
                                                                <?= isset($file['upload_date']) ? ' • ' . date('M d, Y g:i A', strtotime($file['upload_date'])) : '' ?>
                                                            </small>
                                                        </div>
                                                        <div class="btn-group btn-group-sm" role="group">
                                                            <a href="download_teacher_feedback.php?file=<?= urlencode($file['file_path']) ?>&submission=<?= $submission['submission_id'] ?>" 
                                                               class="btn btn-outline-success btn-sm me-1" title="Download File">
                                                                <i class="bx bx-download"></i> Download
                                                            </a>
                                                            <button type="button" class="btn btn-outline-danger btn-sm" 
                                                                    onclick="deleteTeacherFile('<?= htmlspecialchars($file['file_path']) ?>', <?= $submission['submission_id'] ?>)"
                                                                    title="Delete File">
                                                                <i class="bx bx-trash"></i> Delete
                                                            </button>
                                                        </div>
                                                    </div>
                                                    
                                                    <!-- File Preview Area -->
                                                    <div id="file_preview_<?= $submission['submission_id'] ?>_<?= $index ?>" class="file-preview-area mt-2" style="display: none;">
                                                        <!-- Preview content will be loaded here -->
                                                    </div>
                                                </div>
                                            <?php endforeach; ?>
                                        </div>
                                    <?php endif; ?>
                                    
                                    <!-- File Upload Input -->
                                    <div class="file-upload-section">
                                        <input type="file" 
                                               name="teacher_feedback_files[]" 
                                               id="teacher_files_<?= $submission['student_id'] ?>"
                                               class="form-control teacher-file-input" 
                                               multiple 
                                               accept=".pdf,.doc,.docx,.txt,.jpg,.jpeg,.png,.gif,.mp4,.mov,.avi,.mp3,.wav,.xlsx,.xls,.ppt,.pptx"
                                               onchange="previewTeacherFiles(this, <?= $submission['student_id'] ?>)">
                                        <div class="form-text">
                                            <small class="text-muted">
                                                <i class="bx bx-info-circle"></i>
                                                Supported: PDF, Documents, Images, Videos, Audio, Spreadsheets, Presentations (Max 10MB each)
                                            </small>
                                        </div>
                                        
                                        <!-- Preview Container -->
                                        <div id="preview_container_<?= $submission['student_id'] ?>" class="file-previews mt-3" style="display: none;">
                                            <h6 class="text-primary"><i class="bx bx-show"></i> File Previews:</h6>
                                            <div id="preview_content_<?= $submission['student_id'] ?>" class="preview-grid">
                                                <!-- Previews will be inserted here -->
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="form-group">
                                    <button type="submit" class="btn btn-primary">
                                        <i class="bx bx-save"></i>
                                        Update Grade & Files
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
    </div>
</div>
</div>

<style>
/* File Preview Styling */
.file-previews {
    background: #f8f9fa;
    border: 1px solid #e9ecef;
    border-radius: 8px;
    padding: 15px;
}

.preview-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 15px;
    margin-top: 10px;
}

.preview-item {
    background: white;
    border: 1px solid #dee2e6;
    border-radius: 8px;
    overflow: hidden;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
    position: relative;
}

.preview-header {
    background: #007bff;
    color: white;
    padding: 8px 12px;
    font-size: 0.9rem;
    font-weight: 500;
    text-align: center;
}

.preview-content {
    padding: 10px;
    text-align: center;
}

.preview-image {
    max-width: 100%;
    max-height: 150px;
    border-radius: 4px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
}

.preview-video {
    width: 100%;
    max-height: 150px;
    border-radius: 4px;
}

.preview-file-icon {
    font-size: 3rem;
    color: #6c757d;
    margin-bottom: 10px;
}

.preview-file-info {
    color: #6c757d;
    font-size: 0.8rem;
    margin-top: 5px;
}

.preview-remove {
    position: absolute;
    top: 5px;
    right: 5px;
    background: rgba(220, 53, 69, 0.9);
    color: white;
    border: none;
    border-radius: 50%;
    width: 25px;
    height: 25px;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 14px;
}

.preview-remove:hover {
    background: rgba(220, 53, 69, 1);
}
</style>

<script>
function previewTeacherFiles(input, studentId) {
    const previewContainer = document.getElementById('preview_container_' + studentId);
    const previewContent = document.getElementById('preview_content_' + studentId);
    
    // Clear previous previews
    previewContent.innerHTML = '';
    
    if (input.files && input.files.length > 0) {
        previewContainer.style.display = 'block';
        
        Array.from(input.files).forEach((file, index) => {
            const previewItem = document.createElement('div');
            previewItem.className = 'preview-item';
            previewItem.dataset.fileIndex = index;
            
            const header = document.createElement('div');
            header.className = 'preview-header';
            header.textContent = file.name;
            
            const content = document.createElement('div');
            content.className = 'preview-content';
            
            const removeBtn = document.createElement('button');
            removeBtn.className = 'preview-remove';
            removeBtn.innerHTML = '×';
            removeBtn.onclick = () => removePreviewFile(input, studentId, index);
            
            // Create preview based on file type
            if (file.type.startsWith('image/')) {
                const img = document.createElement('img');
                img.className = 'preview-image';
                img.onload = function() {
                    URL.revokeObjectURL(this.src);
                };
                img.src = URL.createObjectURL(file);
                content.appendChild(img);
                
            } else if (file.type.startsWith('video/')) {
                const video = document.createElement('video');
                video.className = 'preview-video';
                video.controls = true;
                video.muted = true;
                video.src = URL.createObjectURL(file);
                content.appendChild(video);
                
            } else {
                // Show file icon for other types
                const icon = document.createElement('i');
                icon.className = 'bx preview-file-icon';
                
                // Set icon based on file extension
                const ext = file.name.split('.').pop().toLowerCase();
                switch(ext) {
                    case 'pdf':
                        icon.classList.add('bx-file-blank');
                        icon.style.color = '#dc3545';
                        break;
                    case 'doc':
                    case 'docx':
                        icon.classList.add('bx-file-doc');
                        icon.style.color = '#0d6efd';
                        break;
                    case 'xls':
                    case 'xlsx':
                        icon.classList.add('bx-spreadsheet');
                        icon.style.color = '#198754';
                        break;
                    case 'ppt':
                    case 'pptx':
                        icon.classList.add('bx-slideshow');
                        icon.style.color = '#fd7e14';
                        break;
                    case 'mp3':
                    case 'wav':
                        icon.classList.add('bx-music');
                        icon.style.color = '#6f42c1';
                        break;
                    default:
                        icon.classList.add('bx-file');
                        break;
                }
                
                content.appendChild(icon);
            }
            
            // Add file size info
            const fileInfo = document.createElement('div');
            fileInfo.className = 'preview-file-info';
            fileInfo.textContent = formatFileSize(file.size);
            content.appendChild(fileInfo);
            
            previewItem.appendChild(header);
            previewItem.appendChild(content);
            previewItem.appendChild(removeBtn);
            previewContent.appendChild(previewItem);
        });
    } else {
        previewContainer.style.display = 'none';
    }
}

function removePreviewFile(input, studentId, indexToRemove) {
    const dt = new DataTransfer();
    
    // Add all files except the one to remove
    Array.from(input.files).forEach((file, index) => {
        if (index !== indexToRemove) {
            dt.items.add(file);
        }
    });
    
    // Update input files
    input.files = dt.files;
    
    // Refresh previews
    previewTeacherFiles(input, studentId);
}

function formatFileSize(bytes) {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
}

function viewTeacherFile(filePath, fileName, fileExt, submissionId) {
    const previewId = 'file_preview_' + submissionId + '_' + Math.floor(Math.random() * 1000);
    const previewArea = document.querySelector(`[id*="file_preview_${submissionId}_"]`);
    
    if (previewArea.style.display === 'block') {
        previewArea.style.display = 'none';
        return;
    }
    
    previewArea.style.display = 'block';
    previewArea.innerHTML = '<div class="text-center p-3"><i class="bx bx-loader-alt bx-spin"></i> Loading...</div>';
    
    let previewContent = '';
    
    if (['jpg', 'jpeg', 'png', 'gif'].includes(fileExt)) {
        previewContent = `
            <div class="text-center p-3">
                <img src="download_teacher_feedback.php?file=${encodeURIComponent(filePath)}&submission=${submissionId}&inline=1" 
                     class="img-fluid" style="max-height: 400px; border-radius: 8px; box-shadow: 0 4px 8px rgba(0,0,0,0.1);"
                     onerror="this.parentElement.innerHTML='<div class=\\'alert alert-danger\\'>Error loading image. <a href=\\'download_teacher_feedback.php?file=${encodeURIComponent(filePath)}&submission=${submissionId}\\' target=\\'_blank\\'>Download instead</a></div>'">
            </div>`;
    } else if (['mp4', 'mov', 'avi', 'webm'].includes(fileExt)) {
        previewContent = `
            <div class="text-center p-3">
                <video controls style="max-width: 100%; max-height: 400px; border-radius: 8px;">
                    <source src="download_teacher_feedback.php?file=${encodeURIComponent(filePath)}&submission=${submissionId}&inline=1" type="video/${fileExt}">
                    Your browser does not support the video tag.
                </video>
            </div>`;
    } else if (fileExt === 'pdf') {
        previewContent = `
            <div class="p-3">
                <div class="alert alert-info mb-3">
                    <i class="bx bx-info-circle"></i> PDF Preview - <strong>${fileName}</strong>
                </div>
                <iframe src="download_teacher_feedback.php?file=${encodeURIComponent(filePath)}&submission=${submissionId}&inline=1" 
                        width="100%" height="500" style="border: 1px solid #ddd; border-radius: 8px;">
                    <p>Your browser does not support PDFs. 
                       <a href="download_teacher_feedback.php?file=${encodeURIComponent(filePath)}&submission=${submissionId}">Download the PDF</a>.
                    </p>
                </iframe>
            </div>`;
    } else if (['mp3', 'wav'].includes(fileExt)) {
        previewContent = `
            <div class="text-center p-3">
                <audio controls style="width: 100%;">
                    <source src="download_teacher_feedback.php?file=${encodeURIComponent(filePath)}&submission=${submissionId}&inline=1" type="audio/${fileExt}">
                    Your browser does not support the audio element.
                </audio>
            </div>`;
    } else {
        previewContent = `
            <div class="text-center p-3">
                <div class="alert alert-warning">
                    <i class="bx bx-file"></i> Preview not available for this file type.<br>
                    <a href="download_teacher_feedback.php?file=${encodeURIComponent(filePath)}&submission=${submissionId}" 
                       class="btn btn-primary mt-2">
                        <i class="bx bx-download"></i> Download ${fileName}
                    </a>
                </div>
            </div>`;
    }
    
    previewArea.innerHTML = previewContent;
}

async function deleteTeacherFile(submissionId, fileIndex, fileName) {
    if (!confirm(`Are you sure you want to delete "${fileName}"? This action cannot be undone.`)) {
        return;
    }
    
    try {
        const response = await fetch('delete_teacher_feedback.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                submission_id: submissionId,
                file_index: fileIndex
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            // Reload the page to refresh the file list
            location.reload();
        } else {
            alert('Error deleting file: ' + (result.message || 'Unknown error'));
        }
    } catch (error) {
        alert('Error deleting file: ' + error.message);
    }
}

function removeTeacherFile(studentId, fileIndex) {
    if (confirm('Are you sure you want to remove this teacher feedback file?')) {
        // This would need AJAX implementation to remove from database
        alert('File removal functionality needs AJAX implementation');
    }
}
</script>
