<?php
/**
 * View Video Page - Teacher Portal
 * 
 * This page allows teachers to view and manage their uploaded videos, including
 * video playback, publishing/unpublishing, duplication, and viewing similar videos.
 */

// Set current page for active menu highlighting
$current_page = 'videos.php';

// Include the centralized teacher layout
require_once "components/teacher_layout.php";

// Enable error reporting during development
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Ensure mysqli connection is available for legacy code
if (!isset($conn) && isset($pdo)) {
    $conn = new mysqli($host, $username, $password, $dbname);
    if ($conn->connect_error) {
        die("Connection failed: " . $conn->connect_error);
    }
}

/**
 * Get teacher data with robust fallback approach
 */
function get_teacher($conn, $teacher_id) {
    // Default teacher data in case of failure
    $default_teacher = [
        'user_id' => $teacher_id,
        'first_name' => 'Academy',
        'last_name' => 'Teacher',
        'email' => 'teacher@panadite.com',
        'profile_image' => null,
        'bio' => 'No bio available',
        'created_at' => date('Y-m-d H:i:s')
    ];
    
    try {
        $stmt = $conn->prepare("SELECT * FROM teachers WHERE user_id = ? LIMIT 1");
        if (!$stmt) {
            error_log("Failed to prepare teacher statement: " . $conn->error);
            return $default_teacher;
        }
        
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result && $result->num_rows > 0) {
            return $result->fetch_assoc();
        } else {
            error_log("No teacher found with ID: $teacher_id");
            return $default_teacher;
        }
    } catch (Exception $e) {
        error_log("Error fetching teacher data: " . $e->getMessage());
        return $default_teacher;
    }
}

// Initialize teacher variable
$teacher_id = $_SESSION['user_id'];
$teacher = null;

// Check if database connection is available
if ($conn) {
    try {
        $teacher = get_teacher($conn, $teacher_id);
    } catch (Exception $e) {
        error_log("Error getting teacher info in view_video.php: " . $e->getMessage());
    }
} else {
    error_log("Database connection not available in view_video.php");
}

// Enable error reporting during development
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Ensure mysqli connection is available for legacy code
if (!isset($conn) && isset($pdo)) {
    $conn = new mysqli($host, $username, $password, $dbname);
    if ($conn->connect_error) {
        die("Connection failed: " . $conn->connect_error);
    }
}

// Check if video ID is provided
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    header('Location: videos.php');
    exit;
}

$video_id = (int)$_GET['id'];
$message = '';

// Handle toggle publish
if (isset($_GET['action']) && $_GET['action'] === 'toggle_publish') {
    // Verify ownership through course ownership
    $stmt = $conn->prepare("SELECT v.is_published FROM videos v JOIN courses c ON v.course_id = c.course_id WHERE v.video_id = ? AND c.teacher_id = ?");
    $stmt->bind_param("ii", $video_id, $teacher_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $video_data = $result->fetch_assoc();
    
    if ($result->num_rows > 0) {
        $new_status = $video_data['is_published'] ? 0 : 1;
        $status_text = $new_status ? "published" : "unpublished";
        
        $stmt = $conn->prepare("UPDATE videos SET is_published = ? WHERE video_id = ?");
        $stmt->bind_param("ii", $new_status, $video_id);
        
        if ($stmt->execute()) {
            $message = "<div class='alert alert-success'>Video $status_text successfully.</div>";
        } else {
            $message = "<div class='alert alert-danger'>Error updating video status.</div>";
        }
    }
}

// Handle duplicate video
if (isset($_GET['action']) && $_GET['action'] === 'duplicate') {
    $stmt = $conn->prepare("SELECT v.* FROM videos v JOIN courses c ON v.course_id = c.course_id WHERE v.video_id = ? AND c.teacher_id = ?");
    $stmt->bind_param("ii", $video_id, $teacher_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        $video_to_duplicate = $result->fetch_assoc();
        
        // Create a new title with (Copy) appended
        $new_title = $video_to_duplicate['title'] . ' (Copy)';
        
        $stmt = $conn->prepare("INSERT INTO videos 
            (teacher_id, course_id, title, description, video_url, thumbnail, duration, is_published, created_at) 
            VALUES (?, ?, ?, ?, ?, ?, ?, 0, NOW())");
        
        $stmt->bind_param("iisssssi", 
            $teacher_id, 
            $video_to_duplicate['course_id'],
            $new_title,
            $video_to_duplicate['description'],
            $video_to_duplicate['video_url'],
            $video_to_duplicate['thumbnail'],
            $video_to_duplicate['duration'],
            $video_to_duplicate['is_published']
        );
        
        if ($stmt->execute()) {
            $new_video_id = $conn->insert_id;
            $message = "<div class='alert alert-success'>Video duplicated successfully.</div>";
        } else {
            $message = "<div class='alert alert-danger'>Error duplicating video: " . $conn->error . "</div>";
        }
    }
}

// Verify this video belongs to the logged-in teacher through course ownership
$stmt = $conn->prepare("
    SELECT v.*, c.title as course_title 
    FROM videos v
    LEFT JOIN courses c ON v.course_id = c.course_id
    WHERE v.video_id = ? AND c.teacher_id = ?
");
$stmt->bind_param("ii", $video_id, $teacher_id);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows === 0) {
    // Video not found or doesn't belong to this teacher
    $_SESSION['error_message'] = "Video not found or you don't have permission to view it.";
    header('Location: videos.php');
    exit;
}

$video = $result->fetch_assoc();

// Update view count
$stmt = $conn->prepare("UPDATE videos SET views = views + 1 WHERE video_id = ?");
$stmt->bind_param("i", $video_id);
$stmt->execute();
?>

<!-- Additional styles for video player page -->

<style>
.video-player-container {
    width: 100%;
    border-radius: 8px;
    overflow: hidden;
    background: transparent;
}

.video-player-container.has-video {
    background: #000;
}

.video-player {
    width: 100%;
    height: 400px;
    display: block;
}

.video-placeholder {
    background: #f8f9fa !important;
    border: 2px dashed #dee2e6 !important;
    border-radius: 8px !important;
    min-height: 300px !important;
    display: flex !important;
    align-items: center !important;
    justify-content: center !important;
    flex-direction: column !important;
    padding: 2rem !important;
    margin: 0 !important;
    width: 100% !important;
}

.video-placeholder i {
    margin-bottom: 1rem !important;
    opacity: 0.6 !important;
    color: #007bff !important;
}

.video-placeholder h5 {
    margin-bottom: 0.5rem !important;
    font-weight: 600 !important;
    color: #495057 !important;
}

.video-placeholder p {
    margin-bottom: 1.5rem !important;
    max-width: 300px !important;
    text-align: center !important;
    color: #6c757d !important;
}
</style>

<!-- Page header with title and breadcrumb -->
<div class="page-header">
    <div class="container-fluid px-4">
        <h1 class="page-title"><?php echo htmlspecialchars($video['title']); ?></h1>
        <nav aria-label="breadcrumb">
            <ol class="breadcrumb">
                <li class="breadcrumb-item"><a href="dashboard.php">Home</a></li>
                <li class="breadcrumb-item"><a href="videos.php">Videos</a></li>
                <li class="breadcrumb-item active" aria-current="page">View Video</li>
            </ol>
        </nav>
    </div>
</div>

<!-- Display any messages/alerts -->
<?php if (!empty($message)) : ?>
<div class="alert-container mb-4">
    <?php echo $message; ?>
</div>
<?php endif; ?>

<!-- Main content container with proper spacing -->
<div class="container-fluid px-4">
<div class="row mb-4">
    <div class="col-lg-8">


        <!-- Video Player -->
        <div class="card shadow-sm border-0 mb-4">
            <div class="card-body p-0">
                <?php
                // Simple, clean video display logic
                if ($video['video_type'] == 'url' && $video['platform'] == 'youtube' && !empty($video['video_id_external'])) {
                    // YouTube video
                    ?>
                    <div class="ratio ratio-16x9">
                        <iframe src="https://www.youtube.com/embed/<?php echo htmlspecialchars($video['video_id_external']); ?>?rel=0" 
                                frameborder="0" allowfullscreen 
                                allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture"></iframe>
                    </div>
                    <?php
                } 
                elseif ($video['video_type'] == 'url' && $video['platform'] == 'vimeo' && !empty($video['video_id_external'])) {
                    // Vimeo video
                    ?>
                    <div class="ratio ratio-16x9">
                        <iframe src="https://player.vimeo.com/video/<?php echo htmlspecialchars($video['video_id_external']); ?>" 
                                frameborder="0" allowfullscreen></iframe>
                    </div>
                    <?php
                }
                elseif ($video['video_type'] == 'file' && !empty($video['video_url'])) {
                    // File video - try both relative and absolute paths
                    $video_path = $video['video_url'];
                    if (strpos($video_path, 'uploads/') === 0) {
                        $video_path = '../' . $video_path;
                    }
                    ?>
                    <div class="ratio ratio-16x9" style="background: #000;">
                        <video controls preload="metadata" style="width: 100%; height: 100%;">
                            <source src="<?php echo htmlspecialchars($video_path); ?>" type="video/mp4">
                            <p>Your browser doesn't support HTML5 video. 
                               <a href="<?php echo htmlspecialchars($video_path); ?>">Download the video</a> instead.</p>
                        </video>
                    </div>
                    <?php
                }
                else {
                    // No video or error state
                    ?>
                    <div class="text-center p-5" style="background: #f8f9fa; min-height: 300px; display: flex; align-items: center; justify-content: center; flex-direction: column;">
                        <i class="fas fa-video fa-3x text-muted mb-3"></i>
                        <h5>No Video Available</h5>
                        <p class="text-muted">This video cannot be displayed.</p>
                        <a href="edit_video.php?id=<?php echo $video_id; ?>" class="btn btn-primary">
                            <i class="fas fa-edit me-2"></i>Edit Video
                        </a>
                    </div>
                    <?php
                }
                ?>
            </div>
        </div>

        <!-- Video Details -->
        <div class="card video-details border-0 shadow-sm mb-4">
            <div class="card-body">
                <div class="d-flex justify-content-between align-items-start mb-3">
                    <h1 class="video-title mb-0"><?php echo htmlspecialchars($video['title']); ?></h1>
                    
                    <div class="video-status ms-2">
                        <?php if ($video['is_published']): ?>
                        <span class="badge bg-success">Published</span>
                        <?php else: ?>
                        <span class="badge bg-warning text-dark">Unpublished</span>
                        <?php endif; ?>
                    </div>
                </div>
                
                <div class="meta-info row g-3">
                    <div class="col-auto">
                        <div class="d-flex align-items-center">
                            <div class="meta-icon rounded-circle bg-primary-light me-2">
                                <i class="fas fa-folder text-primary"></i>
                            </div>
                            <div>
                                <div class="text-muted small">Course</div>
                                <div class="fw-medium"><?php echo $video['course_title'] ? htmlspecialchars($video['course_title']) : 'Not Assigned'; ?></div>
                            </div>
                        </div>
                    </div>
                    

                    
                    <div class="col-auto">
                        <div class="d-flex align-items-center">
                            <div class="meta-icon rounded-circle bg-info-light me-2">
                                <i class="fas fa-clock text-info"></i>
                            </div>
                            <div>
                                <div class="text-muted small">Duration</div>
                                <div class="fw-medium"><?php 
                                    // Format duration nicely
                                    $duration = $video['duration'] ?? 0;
                                    
                                    // Check if duration is already formatted as MM:SS
                                    if (is_string($duration) && preg_match('/^(\d+):(\d+)$/', $duration, $matches)) {
                                        // Already formatted, just display it
                                        echo htmlspecialchars($duration);
                                    } else {
                                        // Convert numeric seconds to MM:SS format
                                        $duration = intval($duration); // Ensure it's an integer
                                        $minutes = floor($duration / 60);
                                        $seconds = $duration % 60;
                                        echo $minutes . ':' . str_pad($seconds, 2, '0', STR_PAD_LEFT);
                                    }
                                ?></div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-auto">
                        <div class="d-flex align-items-center">
                            <div class="meta-icon rounded-circle bg-warning-light me-2">
                                <i class="fas fa-calendar-alt text-warning"></i>
                            </div>
                            <div>
                                <div class="text-muted small">Uploaded</div>
                                <div class="fw-medium"><?php echo date('M d, Y', strtotime($video['created_at'])); ?></div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="video-description mt-4">
                    <h5>Description</h5>
                    <div class="text-secondary">
                        <?php 
                        if (!empty($video['description'])) {
                            echo nl2br(htmlspecialchars($video['description'])); 
                        } else {
                            echo '<p class="text-muted fst-italic">No description provided.</p>';
                        }
                        ?>
                    </div>
                </div>
                
                <div class="video-actions mt-4">
                    <div class="d-flex flex-wrap gap-2">
                        <a href="edit_video.php?id=<?php echo $video['video_id']; ?>" class="btn btn-primary">
                            <i class="fas fa-edit me-2"></i> Edit Video
                        </a>
                        

                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <div class="col-lg-4">
        <!-- Similar Videos Section -->
        <div class="card border-0 shadow-sm similar-videos-section">
            <div class="card-header bg-white border-0 py-3">
                <div class="d-flex justify-content-between align-items-center">
                    <h3 class="card-title mb-0 h5"><i class="fas fa-video me-2 text-primary"></i>More Videos From This Course</h3>
                    <a href="videos.php?course_id=<?php echo $video['course_id'] ?? 0; ?>" class="btn btn-sm btn-outline-primary">View All</a>
                </div>
            </div>
            
            <div class="card-body p-0">
                <?php
                // Get other videos from the same course
                $course_id = $video['course_id'] ?? 0;
                if ($course_id > 0) {
                    $stmt = $conn->prepare("
                        SELECT video_id, title, video_url, thumbnail, duration, views 
                        FROM videos 
                        WHERE course_id = ? AND video_id != ? AND teacher_id = ? AND is_published = 1
                        ORDER BY created_at DESC 
                        LIMIT 5
                    ");
                    $stmt->bind_param("iii", $course_id, $video_id, $teacher_id);
                    $stmt->execute();
                    $similar_videos = $stmt->get_result();
                    
                    if ($similar_videos->num_rows > 0) {
                        echo '<ul class="list-group list-group-flush">';
                        while ($similar = $similar_videos->fetch_assoc()) {
                            ?>
                            <li class="list-group-item px-3 py-3 similar-video-item">
                                <div class="d-flex">
                                    <div class="similar-video-thumbnail me-3">
                                        <img src="<?php 
                                        // Handle all possible thumbnail path formats
                                        if (isset($similar['thumbnail']) && !empty($similar['thumbnail'])) {
                                            if (strpos($similar['thumbnail'], 'http') === 0) {
                                                echo htmlspecialchars($similar['thumbnail']);
                                            } else {
                                                echo htmlspecialchars('../' . $similar['thumbnail']);
                                            }
                                        } else {
                                            echo '../assets/images/video-placeholder.jpg';
                                        }
                                        ?>" alt="<?php echo htmlspecialchars($similar['title'] ?? 'Video thumbnail'); ?>" class="img-fluid rounded" style="width: 100px; height: 60px; object-fit: cover;">
                                        <span class="duration-badge">
                                            <?php 
                                            $dur = $similar['duration'] ?? 0;
                                            $min = floor($dur / 60);
                                            $sec = $dur % 60;
                                            echo $min . ':' . str_pad($sec, 2, '0', STR_PAD_LEFT);
                                            ?>
                                        </span>
                                    </div>
                                    <div class="similar-video-details d-flex flex-column justify-content-between flex-grow-1">
                                        <h5 class="mb-2 h6">
                                            <a href="view_video.php?id=<?php echo $similar['video_id']; ?>" class="text-decoration-none text-dark">
                                                <?php echo htmlspecialchars($similar['title'] ?? 'Untitled Video'); ?>
                                            </a>
                                        </h5>
                                        <div class="d-flex align-items-center">
                                            <span class="text-muted small me-3">
                                                <i class="fas fa-eye me-1"></i> <?php echo number_format($similar['views'] ?? 0); ?>
                                            </span>
                                        </div>
                                    </div>
                                </div>
                            </li>
                            <?php
                        }
                        echo '</ul>';
                    } else {
                        echo '<div class="p-4 text-center">
                                <div class="text-muted mb-3"><i class="fas fa-video fa-2x"></i></div>
                                <p>No other videos found in this course.</p>
                                <a href="upload-video.php?course_id=' . $course_id . '" class="btn btn-sm btn-primary">Upload New Video</a>
                            </div>';
                    }
                } else {
                    echo '<div class="p-4 text-center">
                            <div class="text-muted mb-3"><i class="fas fa-exclamation-circle fa-2x"></i></div>
                            <p>This video is not assigned to any course.</p>
                            <a href="edit_video.php?id=' . $video_id . '" class="btn btn-sm btn-primary">Assign to Course</a>
                        </div>';
                }
                ?>
            </div>
        </div>
    </div>
</div>
</div>
<!-- End container with proper spacing -->

<!-- JavaScript for video functionality -->
<script>
    document.addEventListener('DOMContentLoaded', () => {
        // Copy share link functionality
        window.copyShareLink = function() {
            const shareUrl = window.location.origin + '/watch.php?id=<?php echo $video_id; ?>';
            navigator.clipboard.writeText(shareUrl).then(() => {
                alert('Share link copied to clipboard!');
            }).catch(err => {
                console.error('Failed to copy: ', err);
            });
        }
    });
</script>

<?php
require_once "components/teacher_layout_end.php";
?>
                    