<?php
require_once __DIR__ . '/../auth/functions.php';
require_once __DIR__ . '/../includes/config.php';

// Ensure user is logged in as an admin
requireRole('admin');

// Handle status updates
if ($_POST && isset($_POST['update_status'])) {
    $request_id = $_POST['request_id'];
    $new_status = $_POST['status'];
    $admin_notes = $_POST['admin_notes'];
    
    $update_query = "UPDATE withdrawal_requests SET 
                    status = ?, 
                    admin_notes = ?, 
                    processed_by = ?, 
                    processed_at = NOW() 
                    WHERE id = ?";
    $stmt = $conn->prepare($update_query);
    $stmt->bind_param("ssii", $new_status, $admin_notes, $_SESSION['user_id'], $request_id);
    $stmt->execute();
}

// Get filter parameters
$status_filter = $_GET['status'] ?? '';
$teacher_filter = $_GET['teacher'] ?? '';

// Build query with filters
$query = "SELECT wr.*, u.username, u.email 
          FROM withdrawal_requests wr 
          JOIN users u ON wr.teacher_id = u.user_id 
          WHERE 1=1";

$params = [];
$types = "";

if ($status_filter) {
    $query .= " AND wr.status = ?";
    $params[] = $status_filter;
    $types .= "s";
}

if ($teacher_filter) {
    $query .= " AND wr.teacher_id = ?";
    $params[] = $teacher_filter;
    $types .= "i";
}

$query .= " ORDER BY wr.requested_at DESC";

$stmt = $conn->prepare($query);
if ($params) {
    $stmt->bind_param($types, ...$params);
}
$stmt->execute();
$requests = $stmt->get_result();

// Get summary statistics
$stats_query = "SELECT 
                COUNT(*) as total_requests,
                COUNT(CASE WHEN status = 'pending' THEN 1 END) as pending,
                COUNT(CASE WHEN status = 'approved' THEN 1 END) as approved,
                COUNT(CASE WHEN status = 'completed' THEN 1 END) as completed,
                COUNT(CASE WHEN status = 'rejected' THEN 1 END) as rejected,
                SUM(CASE WHEN status IN ('approved', 'completed') THEN amount ELSE 0 END) as total_approved
                FROM withdrawal_requests";
$stats = $conn->query($stats_query)->fetch_assoc();

// Get teachers list for filter
$teachers_query = "SELECT DISTINCT u.user_id, u.username 
                  FROM users u 
                  JOIN withdrawal_requests wr ON u.user_id = wr.teacher_id 
                  ORDER BY u.username";
$teachers = $conn->query($teachers_query);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Withdrawal Requests Management - Admin Panel</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link rel="stylesheet" href="/assets/css/user-cards.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        /* Modern Panadite Design System */
        :root {
            --primary-color: #4361ee;
            --secondary-color: #3f37c9;
            --success-color: #10b981;
            --warning-color: #f59e0b;
            --danger-color: #ef4444;
            --info-color: #3b82f6;
            --dark-color: #1f2937;
            --light-bg: #f8fafc;
            --white: #ffffff;
            --border-color: #e2e8f0;
            --text-muted: #64748b;
            --shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
            --shadow-lg: 0 10px 15px -3px rgba(0, 0, 0, 0.1);
        }

        * {
            font-family: 'Inter', 'Poppins', sans-serif !important;
        }

        body {
            background: var(--light-bg) !important;
            color: var(--dark-color) !important;
        }

        /* Modern Page Header */
        .page-header {
            background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%) !important;
            color: white !important;
            padding: 2rem !important;
            border-radius: 16px !important;
            margin-bottom: 2rem !important;
            box-shadow: var(--shadow-lg) !important;
        }

        .page-title {
            font-size: 2rem !important;
            font-weight: 700 !important;
            margin-bottom: 0.5rem !important;
            display: flex !important;
            align-items: center !important;
            gap: 12px !important;
        }

        .page-description {
            font-size: 1rem !important;
            opacity: 0.9 !important;
            margin: 0 !important;
        }

        /* Modern Stats Cards */
        .stats-container {
            display: grid !important;
            grid-template-columns: repeat(auto-fit, minmax(180px, 1fr)) !important;
            gap: 20px !important;
            margin-bottom: 32px !important;
        }

        .modern-stat-card {
            background: var(--white) !important;
            border-radius: 16px !important;
            padding: 24px !important;
            box-shadow: var(--shadow) !important;
            border: 1px solid var(--border-color) !important;
            transition: all 0.3s ease !important;
            position: relative !important;
            overflow: hidden !important;
        }

        .modern-stat-card:hover {
            transform: translateY(-2px) !important;
            box-shadow: var(--shadow-lg) !important;
        }

        .modern-stat-card::before {
            content: '' !important;
            position: absolute !important;
            top: 0 !important;
            left: 0 !important;
            right: 0 !important;
            height: 4px !important;
            background: var(--primary-color) !important;
        }

        .stat-header {
            display: flex !important;
            align-items: center !important;
            justify-content: space-between !important;
            margin-bottom: 16px !important;
        }

        .stat-icon {
            width: 48px !important;
            height: 48px !important;
            border-radius: 12px !important;
            display: flex !important;
            align-items: center !important;
            justify-content: center !important;
            font-size: 20px !important;
            color: white !important;
            background: var(--primary-color) !important;
        }

        .stat-number {
            font-size: 2rem !important;
            font-weight: 800 !important;
            color: var(--dark-color) !important;
            line-height: 1 !important;
            margin-bottom: 8px !important;
        }

        .stat-label {
            font-size: 0.875rem !important;
            color: var(--text-muted) !important;
            font-weight: 500 !important;
            margin: 0 !important;
        }

        /* Color Variants */
        .modern-stat-card.success::before { background: var(--success-color) !important; }
        .modern-stat-card.success .stat-icon { background: var(--success-color) !important; }
        
        .modern-stat-card.warning::before { background: var(--warning-color) !important; }
        .modern-stat-card.warning .stat-icon { background: var(--warning-color) !important; }
        
        .modern-stat-card.danger::before { background: var(--danger-color) !important; }
        .modern-stat-card.danger .stat-icon { background: var(--danger-color) !important; }
        
        .modern-stat-card.info::before { background: var(--info-color) !important; }
        .modern-stat-card.info .stat-icon { background: var(--info-color) !important; }
        
        .stats-card.success::before {
            background: #28a745;
        }
        .stats-card.danger::before {
            background: #dc3545;
        }

        .requests-table {
            background: white;
            border-radius: 10px;
            padding: 1.5rem;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }

        .status-badge {
            padding: 0.5rem 1rem;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: bold;
        }

        .status-pending { background-color: #fff3cd; color: #856404; }
        .status-approved { background-color: #d1ecf1; color: #0c5460; }
        .status-completed { background-color: #d4edda; color: #155724; }
        .status-rejected { background-color: #f8d7da; color: #721c24; }

        .action-buttons .btn {
            margin: 2px;
            padding: 0.25rem 0.5rem;
            font-size: 0.8rem;
        }

        .filter-section {
            background: var(--white) !important;
            padding: 24px !important;
            border-radius: 16px !important;
            margin-bottom: 24px !important;
            box-shadow: var(--shadow) !important;
            border: 1px solid var(--border-color) !important;
        }

        .filter-title {
            font-size: 1.125rem !important;
            font-weight: 600 !important;
            color: var(--dark-color) !important;
            margin-bottom: 16px !important;
            display: flex !important;
            align-items: center !important;
            gap: 8px !important;
        }

        /* Modern Table */
        .modern-table-container {
            background: var(--white) !important;
            border-radius: 16px !important;
            padding: 24px !important;
            box-shadow: var(--shadow) !important;
            border: 1px solid var(--border-color) !important;
            overflow: hidden !important;
        }

        .table-title {
            font-size: 1.25rem !important;
            font-weight: 600 !important;
            color: var(--dark-color) !important;
            margin-bottom: 20px !important;
            display: flex !important;
            align-items: center !important;
            justify-content: space-between !important;
        }

        .table-count {
            background: var(--primary-color) !important;
            color: white !important;
            padding: 4px 12px !important;
            border-radius: 20px !important;
            font-size: 0.875rem !important;
            font-weight: 500 !important;
        }

        .table {
            border: none !important;
            margin-bottom: 0 !important;
        }

        .table thead th {
            background: var(--light-bg) !important;
            border: none !important;
            color: var(--dark-color) !important;
            font-weight: 600 !important;
            font-size: 0.875rem !important;
            text-transform: uppercase !important;
            letter-spacing: 0.5px !important;
            padding: 16px !important;
        }

        .table tbody td {
            padding: 16px !important;
            vertical-align: middle !important;
            border-bottom: 1px solid var(--border-color) !important;
            font-size: 0.875rem !important;
        }

        .table tbody tr:hover {
            background: rgba(67, 97, 238, 0.02) !important;
        }

        /* Modern Status Badges */
        .status-badge {
            padding: 6px 12px !important;
            border-radius: 20px !important;
            font-weight: 600 !important;
            font-size: 0.75rem !important;
            text-transform: uppercase !important;
            letter-spacing: 0.5px !important;
            border: 2px solid transparent !important;
        }

        .status-pending {
            background: rgba(245, 158, 11, 0.1) !important;
            color: var(--warning-color) !important;
            border-color: rgba(245, 158, 11, 0.2) !important;
        }

        .status-approved {
            background: rgba(16, 185, 129, 0.1) !important;
            color: var(--success-color) !important;
            border-color: rgba(16, 185, 129, 0.2) !important;
        }

        .status-completed {
            background: rgba(59, 130, 246, 0.1) !important;
            color: var(--info-color) !important;
            border-color: rgba(59, 130, 246, 0.2) !important;
        }

        .status-rejected {
            background: rgba(239, 68, 68, 0.1) !important;
            color: var(--danger-color) !important;
            border-color: rgba(239, 68, 68, 0.2) !important;
        }

        /* Modern Buttons */
        .btn-modern {
            padding: 8px 16px !important;
            border-radius: 8px !important;
            font-weight: 500 !important;
            font-size: 0.875rem !important;
            border: 2px solid transparent !important;
            transition: all 0.2s ease !important;
        }

        .btn-modern.btn-primary {
            background: var(--primary-color) !important;
            color: white !important;
            border-color: var(--primary-color) !important;
        }

        .btn-modern.btn-success {
            background: var(--success-color) !important;
            color: white !important;
            border-color: var(--success-color) !important;
        }

        .btn-modern.btn-danger {
            background: var(--danger-color) !important;
            color: white !important;
            border-color: var(--danger-color) !important;
        }

        .btn-modern:hover {
            transform: translateY(-1px) !important;
            box-shadow: 0 4px 12px rgba(67, 97, 238, 0.3) !important;
        }

        /* Form Controls */
        .form-select, .form-control {
            border: 2px solid var(--border-color) !important;
            border-radius: 8px !important;
            padding: 12px 16px !important;
            font-size: 0.875rem !important;
            transition: all 0.2s ease !important;
        }

        .form-select:focus, .form-control:focus {
            border-color: var(--primary-color) !important;
            box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.1) !important;
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <?php include 'components/sidebar.php'; ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include 'components/navbar.php'; ?>
        
        <!-- Dashboard Content -->
        <div class="dashboard-content">
            <div class="container-fluid p-4">
                <!-- Modern Page Header -->
                <div class="page-header">
                    <h1 class="page-title">
                        <i class='bx bx-money-withdraw'></i>
                        Withdrawal Requests Management
                    </h1>
                    <p class="page-description">Manage teacher withdrawal requests and payouts</p>
                </div>
                <!-- Modern Statistics Cards -->
                <div class="stats-container">
                    <div class="modern-stat-card">
                        <div class="stat-header">
                            <div class="stat-icon">
                                <i class='bx bx-list-ul'></i>
                            </div>
                        </div>
                        <div class="stat-number"><?php echo $stats['total_requests']; ?></div>
                        <div class="stat-label">Total Requests</div>
                    </div>

                    <div class="modern-stat-card warning">
                        <div class="stat-header">
                            <div class="stat-icon">
                                <i class='bx bx-time-five'></i>
                            </div>
                        </div>
                        <div class="stat-number"><?php echo $stats['pending']; ?></div>
                        <div class="stat-label">Pending</div>
                    </div>

                    <div class="modern-stat-card success">
                        <div class="stat-header">
                            <div class="stat-icon">
                                <i class='bx bx-check-circle'></i>
                            </div>
                        </div>
                        <div class="stat-number"><?php echo $stats['approved']; ?></div>
                        <div class="stat-label">Approved</div>
                    </div>

                    <div class="modern-stat-card info">
                        <div class="stat-header">
                            <div class="stat-icon">
                                <i class='bx bx-check-double'></i>
                            </div>
                        </div>
                        <div class="stat-number"><?php echo $stats['completed']; ?></div>
                        <div class="stat-label">Completed</div>
                    </div>

                    <div class="modern-stat-card danger">
                        <div class="stat-header">
                            <div class="stat-icon">
                                <i class='bx bx-x-circle'></i>
                            </div>
                        </div>
                        <div class="stat-number"><?php echo $stats['rejected']; ?></div>
                        <div class="stat-label">Rejected</div>
                    </div>

                    <div class="modern-stat-card success">
                        <div class="stat-header">
                            <div class="stat-icon">
                                <i class='bx bx-money'></i>
                            </div>
                        </div>
                        <div class="stat-number">R<?php echo number_format($stats['total_approved'], 2); ?></div>
                        <div class="stat-label">Total Approved</div>
                    </div>
                </div>

        <!-- Modern Filters -->
                <div class="filter-section">
                    <div class="filter-title">
                        <i class='bx bx-filter-alt'></i>
                        Filter Requests
                    </div>
                    <form method="GET" class="row g-3">
                        <div class="col-md-3">
                            <label class="form-label">Status</label>
                            <select name="status" class="form-select">
                                <option value="">All Status</option>
                                <option value="pending" <?php echo ($filter_status == 'pending') ? 'selected' : ''; ?>>Pending</option>
                                <option value="approved" <?php echo ($filter_status == 'approved') ? 'selected' : ''; ?>>Approved</option>
                                <option value="completed" <?php echo ($filter_status == 'completed') ? 'selected' : ''; ?>>Completed</option>
                                <option value="rejected" <?php echo ($filter_status == 'rejected') ? 'selected' : ''; ?>>Rejected</option>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <label class="form-label">Teacher</label>
                            <select name="teacher_id" class="form-select">
                                <option value="">All Teachers</option>
                                <?php
                                $teacher_query = "SELECT id, name FROM users WHERE role = 'teacher'";
                                $teacher_result = $conn->query($teacher_query);
                                while ($teacher = $teacher_result->fetch_assoc()) {
                                    $selected = ($filter_teacher_id == $teacher['id']) ? 'selected' : '';
                                    echo "<option value='{$teacher['id']}' {$selected}>{$teacher['name']}</option>";
                                }
                                ?>
                            </select>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">From Date</label>
                            <input type="date" name="date_from" class="form-control" value="<?php echo $filter_date_from; ?>">
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">To Date</label>
                            <input type="date" name="date_to" class="form-control" value="<?php echo $filter_date_to; ?>">
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">&nbsp;</label>
                            <div>
                                <button type="submit" class="btn btn-modern btn-primary w-100">
                                    <i class='bx bx-search'></i> Filter
                                </button>
                            </div>
                        </div>
                    </form>
                </div>

                <!-- Modern Table -->
                <div class="modern-table-container">
                    <div class="table-title">
                        <span>
                            <i class='bx bx-table'></i>
                            Withdrawal Requests
                        </span>
                        <span class="table-count"><?php echo count($requests); ?></span>
                    </div>

                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                            <th>Teacher</th>
                            <th>Amount</th>
                            <th>Payment Method</th>
                            <th>Status</th>
                            <th>Requested</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if ($requests->num_rows > 0): ?>
                            <?php while ($request = $requests->fetch_assoc()): ?>
                                <tr>
                                    <td>#<?php echo $request['id']; ?></td>
                                    <td>
                                        <div class="fw-bold"><?php echo htmlspecialchars($request['username']); ?></div>
                                        <div class="text-muted small"><?php echo htmlspecialchars($request['email']); ?></div>
                                    </td>
                                    <td class="fw-bold">$<?php echo number_format($request['amount'], 2); ?></td>
                                    <td><?php echo ucfirst(str_replace('_', ' ', $request['payment_method'])); ?></td>
                                    <td>
                                        <span class="status-badge status-<?php echo $request['status']; ?>">
                                            <?php echo ucfirst($request['status']); ?>
                                        </span>
                                    </td>
                                    <td><?php echo date('M j, Y g:i A', strtotime($request['requested_at'])); ?></td>
                                    <td class="text-center">
                                        <button class="btn btn-modern btn-primary btn-sm me-2" onclick="viewRequest(<?php echo $request['id']; ?>)" title="View Details">
                                            <i class='bx bx-eye'></i>
                                        </button>
                                        <?php if ($request['status'] == 'pending'): ?>
                                            <button class="btn btn-modern btn-success btn-sm me-2" onclick="updateStatus(<?php echo $request['id']; ?>, 'approved')" title="Approve">
                                                <i class='bx bx-check'></i>
                                            </button>
                                            <button class="btn btn-modern btn-danger btn-sm" onclick="updateStatus(<?php echo $request['id']; ?>, 'rejected')" title="Reject">
                                                <i class='bx bx-x'></i>
                                            </button>
                                        <?php elseif ($request['status'] == 'approved'): ?>
                                            <button class="btn btn-modern btn-success btn-sm" onclick="updateStatus(<?php echo $request['id']; ?>, 'completed')" title="Mark as Completed">
                                                <i class='bx bx-check-double'></i>
                                            </button>
                                        <?php endif; ?>
                                    </td>
                                    </tr>
                                <?php endwhile; ?>
                            <?php else: ?>
                                <tr>
                                    <td colspan="7" class="text-center py-5">
                                        <i class="fas fa-inbox fa-3x text-muted mb-3"></i>
                                        <p class="text-muted">No withdrawal requests found</p>
                                    </td>
                                </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- View Request Modal -->
    <div class="modal fade" id="requestModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-money-bill-wave me-2"></i>
                        Withdrawal Request Details
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body" id="requestDetails">
                    <!-- Content will be populated by JavaScript -->
                </div>
            </div>
        </div>
    </div>

    <!-- Update Status Modal -->
    <div class="modal fade" id="statusModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <form method="POST">
                    <div class="modal-header">
                        <h5 class="modal-title">Update Request Status</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <input type="hidden" name="request_id" id="statusRequestId">
                        <input type="hidden" name="status" id="statusValue">
                        
                        <div class="mb-3">
                            <label class="form-label">New Status</label>
                            <input type="text" class="form-control" id="statusDisplay" readonly>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Admin Notes</label>
                            <textarea name="admin_notes" class="form-control" rows="3" 
                                     placeholder="Add notes about this decision..."></textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" name="update_status" class="btn btn-primary">Update Status</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function viewRequest(request) {
            const modal = new bootstrap.Modal(document.getElementById('requestModal'));
            const detailsHtml = `
                <div class="row">
                    <div class="col-md-6">
                        <h6>Request Information</h6>
                        <p><strong>Request ID:</strong> #${request.id}</p>
                        <p><strong>Amount:</strong> $${parseFloat(request.amount).toLocaleString('en-US', {minimumFractionDigits: 2})}</p>
                        <p><strong>Status:</strong> <span class="status-badge status-${request.status}">${request.status.charAt(0).toUpperCase() + request.status.slice(1)}</span></p>
                        <p><strong>Requested:</strong> ${new Date(request.requested_at).toLocaleDateString()}</p>
                        ${request.processed_at ? `<p><strong>Processed:</strong> ${new Date(request.processed_at).toLocaleDateString()}</p>` : ''}
                    </div>
                    <div class="col-md-6">
                        <h6>Teacher Information</h6>
                        <p><strong>Name:</strong> ${request.username}</p>
                        <p><strong>Email:</strong> ${request.email}</p>
                        <p><strong>Payment Method:</strong> ${request.payment_method.replace('_', ' ').replace(/\b\w/g, l => l.toUpperCase())}</p>
                    </div>
                </div>
                <div class="row">
                    <div class="col-12">
                        <h6>Payment Details</h6>
                        <div class="bg-light p-3 rounded">
                            <pre>${request.payment_details}</pre>
                        </div>
                    </div>
                </div>
                ${request.notes ? `
                <div class="row mt-3">
                    <div class="col-12">
                        <h6>Teacher Notes</h6>
                        <div class="bg-light p-3 rounded">
                            ${request.notes}
                        </div>
                    </div>
                </div>
                ` : ''}
                ${request.admin_notes ? `
                <div class="row mt-3">
                    <div class="col-12">
                        <h6>Admin Notes</h6>
                        <div class="bg-warning p-3 rounded">
                            ${request.admin_notes}
                        </div>
                    </div>
                </div>
                ` : ''}
            `;
            
            document.getElementById('requestDetails').innerHTML = detailsHtml;
            modal.show();
        }

        function updateStatus(requestId, newStatus) {
            document.getElementById('statusRequestId').value = requestId;
            document.getElementById('statusValue').value = newStatus;
            document.getElementById('statusDisplay').value = newStatus.charAt(0).toUpperCase() + newStatus.slice(1);
            
            const modal = new bootstrap.Modal(document.getElementById('statusModal'));
            modal.show();
        }
    </script>
            </div> <!-- container-fluid p-4 -->
        </div> <!-- dashboard-content -->
    </section> <!-- main-content -->
</body>
</html>
