<?php
// Set current page for active menu highlighting
$current_page = 'upload-video.php';

// Include the centralized teacher layout start
require_once __DIR__ . '/components/teacher_layout.php';

// Error handling and reporting during development
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Fatal error handler to capture and display errors
function upload_video_shutdown_handler() {
    $error = error_get_last();
    if ($error !== NULL && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
        echo '<div class="alert alert-danger">';
        echo '<h4>Fatal Error Occurred:</h4>';
        echo '<p>' . $error['message'] . '</p>';
        echo '<p>File: ' . $error['file'] . ' on line ' . $error['line'] . '</p>';
        echo '</div>';
    }
}
register_shutdown_function('upload_video_shutdown_handler');

// Include required files
require_once __DIR__ . '/../includes/functions.php';

// Get teacher data with robust fallback for emergency/test accounts
function upload_video_get_teacher($conn, $user_id) {
    // For emergency login user_id (999)
    if ($user_id == 999) {
        return [
            'user_id' => 999,
            'username' => 'teacher',
            'first_name' => 'Emergency',
            'last_name' => 'Teacher',
            'email' => 'emergency@example.com',
            'bio' => 'Emergency teacher account',
            'qualifications' => 'Emergency credentials',
            'expertise' => 'System testing',
            'status' => 'active'
        ];
    }
    
    // Ensure we have a valid database connection
    if (!$conn || $conn->connect_error) {
        error_log('Database connection not available in upload_video_get_teacher');
        return [
            'user_id' => $user_id,
            'username' => 'db_error_user',
            'first_name' => 'Database', 
            'last_name' => 'Error',
            'email' => 'error@example.com',
            'bio' => 'Database connection error',
            'qualifications' => 'N/A',
            'expertise' => 'N/A',
            'status' => 'active'
        ];
    }
    
    try {
        // First check if the user exists
        $user_check = $conn->query("SELECT * FROM users WHERE user_id = {$user_id}");
        if (!$user_check || $user_check->num_rows == 0) {
            error_log("User with ID {$user_id} not found in upload_video_get_teacher");
            // Return fallback data
            return [
                'user_id' => $user_id,
                'username' => 'unknown',
                'first_name' => 'Unknown',
                'last_name' => 'Teacher',
                'email' => 'unknown@example.com',
                'bio' => 'No data available',
                'qualifications' => 'Not available',
                'expertise' => 'Not available',
                'status' => 'active'
            ];
        }
        
        $stmt = $conn->prepare("SELECT u.*, tp.bio, tp.qualifications, tp.expertise, tp.status 
                               FROM users u 
                               LEFT JOIN teacher_profiles tp ON u.user_id = tp.teacher_id 
                               WHERE u.user_id = ?");
                              
        // Check if prepare was successful
        if ($stmt === false) {
            error_log("Prepare failed: (" . $conn->errno . ") " . $conn->error);
            // Fall back to a simpler query without join
            $stmt = $conn->prepare("SELECT * FROM users WHERE user_id = ?");
            if ($stmt === false) {
                error_log("Even simple prepare failed: (" . $conn->errno . ") " . $conn->error);
                // Return data from the earlier query since prepare failed
                return $user_check->fetch_assoc();
            }
        }
        
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result && $result->num_rows > 0) {
            $teacher = $result->fetch_assoc();
            
            // Add default values for missing fields
            $teacher['bio'] = $teacher['bio'] ?? 'No biography available';
            $teacher['qualifications'] = $teacher['qualifications'] ?? 'Not specified';
            $teacher['expertise'] = $teacher['expertise'] ?? 'Not specified';
            $teacher['status'] = $teacher['status'] ?? 'active';
            
            return $teacher;
        }
    } catch (Exception $e) {
        error_log("Error getting teacher: " . $e->getMessage());
    }
    
    // Fallback data if query fails
    return [
        'user_id' => $user_id,
        'username' => 'unknown',
        'first_name' => 'Unknown',
        'last_name' => 'Teacher',
        'email' => 'unknown@example.com',
        'bio' => 'No data available',
        'qualifications' => 'Not available',
        'expertise' => 'Not available',
        'status' => 'active'
    ];
}

// Get courses with robust error handling
function get_teacher_courses_for_upload($conn, $teacher_id) {
    // For emergency/test account
    if ($teacher_id == 999) {
        return [
            ['course_id' => 101, 'title' => 'Demo Course 1'],
            ['course_id' => 102, 'title' => 'Demo Course 2'],
            ['course_id' => 103, 'title' => 'Demo Course 3']
        ];
    }
    
    // Check database connection
    if (!$conn || $conn->connect_error) {
        error_log("Database connection error in get_teacher_courses_for_upload");
        return [];
    }
    
    try {
        // Check if the courses table exists
        $check_table = $conn->query("SHOW TABLES LIKE 'courses'");
        if ($check_table && $check_table->num_rows == 0) {
            error_log("courses table does not exist in get_teacher_courses_for_upload");
            return [];
        }
        
        // First try to get courses for this specific teacher (including drafts)
        $query = "SELECT course_id, title FROM courses WHERE teacher_id = ? ORDER BY title";
        $stmt = $conn->prepare($query);
        
        // Check if prepare was successful
        if ($stmt === false) {
            error_log("Prepare failed in get_teacher_courses_for_upload: (" . $conn->errno . ") " . $conn->error);
            return [];
        }
        
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result) {
            $courses = $result->fetch_all(MYSQLI_ASSOC);
            $stmt->close();
            
            // If no courses found for this teacher, check if they're an admin or return all courses for development
            if (empty($courses)) {
                error_log("No courses found for teacher_id: $teacher_id, checking user role");
                
                // Check if user is admin or if we should show all courses in development
                $user_check = $conn->prepare("SELECT user_role FROM users WHERE user_id = ?");
                if ($user_check) {
                    $user_check->bind_param("i", $teacher_id);
                    $user_check->execute();
                    $user_result = $user_check->get_result();
                    
                    if ($user_result && $user_result->num_rows > 0) {
                        $user_data = $user_result->fetch_assoc();
                        
                        // If admin or in development mode, show all courses
                        if ($user_data['user_role'] == 'admin' || $teacher_id <= 10) {
                            error_log("User is admin or in development mode, showing all courses");
                            $all_courses_query = "SELECT course_id, title FROM courses WHERE is_published = 1 ORDER BY title";
                            $all_courses_result = $conn->query($all_courses_query);
                            
                            if ($all_courses_result) {
                                $courses = $all_courses_result->fetch_all(MYSQLI_ASSOC);
                            }
                        }
                    }
                    $user_check->close();
                }
            }
            
            return $courses;
        }
    } catch (Exception $e) {
        error_log("Error getting teacher courses: " . $e->getMessage());
    }
    
    return [];
}

// Count functions for navbar badges
function robust_count_teacher_courses($conn, $teacher_id) {
    if ($teacher_id == 999) {
        return 3; // Demo count for emergency account
    }
    
    if (!$conn || $conn->connect_error) {
        return 0;
    }
    
    try {
        // Check if courses table exists
        $check_table = $conn->query("SHOW TABLES LIKE 'courses'");
        if ($check_table && $check_table->num_rows == 0) {
            error_log("courses table does not exist in count_teacher_courses");
            return 0;
        }
        
        // Prepare statement
        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM courses WHERE teacher_id = ?");
        
        // Check if prepare was successful
        if ($stmt === false) {
            error_log("Course count prepare failed: (" . $conn->errno . ") " . $conn->error);
            return 0;
        }
        
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result && $row = $result->fetch_assoc()) {
            return $row['count'];
        }
    } catch (Exception $e) {
        error_log("Error counting courses: " . $e->getMessage());
    }
    
    return 0;
}

function robust_count_teacher_videos($conn, $teacher_id) {
    if ($teacher_id == 999) {
        return 5; // Demo count for emergency account
    }
    
    if (!$conn || $conn->connect_error) {
        return 0;
    }
    
    try {
        // Check if videos table exists
        $check_table = $conn->query("SHOW TABLES LIKE 'videos'");
        if ($check_table && $check_table->num_rows == 0) {
            error_log("videos table does not exist in count_teacher_videos");
            return 0;
        }
        
        // Prepare statement
        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM videos WHERE teacher_id = ?");
        
        // Check if prepare was successful
        if ($stmt === false) {
            error_log("Video count prepare failed: (" . $conn->errno . ") " . $conn->error);
            return 0;
        }
        
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result && $row = $result->fetch_assoc()) {
            return $row['count'];
        }
    } catch (Exception $e) {
        error_log("Error counting videos: " . $e->getMessage());
    }
    
    return 0;
}

// Get teacher data and counts
$teacher_id = $_SESSION['user_id'];
$teacher = upload_video_get_teacher($conn, $teacher_id);
$course_count = robust_count_teacher_courses($conn, $teacher_id);
$video_count = robust_count_teacher_videos($conn, $teacher_id);

// Get courses for dropdown
$courses = get_teacher_courses_for_upload($conn, $teacher_id);

// Debug: Let's see what's happening
error_log("DEBUG: Teacher ID = " . $teacher_id);
error_log("DEBUG: Courses found = " . count($courses));
error_log("DEBUG: Courses data = " . print_r($courses, true));
?>
    <style>
        /* Custom styling for the main content area */
        .main-content {
            flex: 1;
            overflow-y: auto;
            padding: 1.5rem;
            background-color: #f8f9fc;
        }

        /* Sidebar toggle button */
        #sidebarToggle {
            background-color: transparent;
            border: none;
            color: #4e73df;
            font-size: 1.5rem;
        }
        
        /* Modern Upload Area Styling */
        .upload-area {
            border: 2px dashed rgba(0, 123, 255, 0.4);
            border-radius: 16px;
            padding: 40px 20px;
            text-align: center;
            position: relative;
            transition: all 0.3s ease;
            background-color: rgba(0, 123, 255, 0.03);
            cursor: pointer;
        }
        
        .upload-area:hover, .upload-area.highlight {
            border-color: #007bff;
            background-color: rgba(0, 123, 255, 0.08);
            transform: translateY(-2px);
            box-shadow: 0 10px 20px rgba(0, 0, 0, 0.05);
        }
        
        .upload-icon {
            color: #007bff;
            opacity: 0.7;
            transition: all 0.3s ease;
        }
        
        .upload-area:hover .upload-icon {
            opacity: 1;
            transform: scale(1.05);
        }
        
        /* Video Preview Area */
        .video-preview-wrapper {
            background-color: #000;
            border-radius: 12px;
            overflow: hidden;
            box-shadow: 0 8px 16px rgba(0,0,0,0.1);
        }
        
        .file-info {
            border-radius: 12px;
            border: 1px solid rgba(0,0,0,0.1);
            box-shadow: 0 4px 12px rgba(0,0,0,0.04);
        }
        
        /* Form Enhancements */
        .form-control:focus, .form-select:focus {
            border-color: #4e73df;
            box-shadow: 0 0 0 0.25rem rgba(78, 115, 223, 0.25);
        }
        
        /* Card Design */
        .card {
            border: none;
            border-radius: 12px;
            box-shadow: 0 6px 18px rgba(0,0,0,0.06);
            transition: all 0.3s ease;
        }
        
        .card:hover {
            box-shadow: 0 10px 24px rgba(0,0,0,0.1);
        }
        
        .card-header {
            background-color: transparent;
            border-bottom: 1px solid rgba(0,0,0,0.08);
            padding: 1.25rem 1.5rem;
        }
        
        .card-body {
            padding: 1.5rem;
        }
        
        /* Status Badge Styling */
        #uploadStatus {
            padding: 6px 10px;
            font-weight: 500;
            letter-spacing: 0.3px;
        }

        /* Add some responsive padding */
        @media (max-width: 768px) {
            .main-content {
                padding: 1rem;
            }
            
            .upload-area {
                padding: 30px 15px;
            }
        }
        
        /* Page-specific styles for video upload */
        .upload-tips-list li {
            position: relative;
            padding-left: 10px;
            border-radius: 8px;
            padding: 8px 12px;
            transition: all 0.2s ease;
        }
        
        .upload-tips-list li:hover {
            background-color: rgba(0,123,255,0.05);
            transform: translateX(3px);
        }
        
        .upload-tips-list svg {
            flex-shrink: 0;
        }
        .upload-tips-list i {
            margin-right: 10px;
            margin-top: 3px;
        }
        
        /* Media queries for responsiveness */
        @media (max-width: 991px) {
            .sidebar {
                transform: translateX(-100%);
                z-index: 1050;
            }
            
            .content-wrapper {
                margin-left: 0;
                width: 100%;
                max-width: 100%;
            }
            
            .sidebar.show {
                transform: translateX(0);
            }
        }
    </style>
</head>
<!-- Upload Video Page Content -->
<div class="container-xl py-4">
   
                    <!-- Notifications -->
                  
                    <!-- Profile -->
                   

                <div class="row">
                    <div class="col-lg-8">
                        <div class="card">
                                <div class="card-header">
                                <h5 class="card-title mb-0"><i class="fas fa-film me-2 text-primary"></i>Video Upload</h5>
                            </div>
                            <div class="card-body">
                                <form id="videoUploadForm" method="post" action="process-video-with-content.php" enctype="multipart/form-data" class="needs-validation" novalidate>
                                    <!-- Hidden fields for form identification and security -->
                                    <input type="hidden" name="form_type" value="video_upload">
                                    <input type="hidden" name="teacher_id" value="<?php echo $teacher_id; ?>">
                                    <!-- Video Type Selection -->
                                    <div class="mb-4">
                                        <label class="form-label fw-bold">Video Source</label>
                                        <div class="btn-group w-100 mb-3" role="group" aria-label="Video source type">
                                            <input type="radio" class="btn-check" name="video_type" id="videoTypeFile" value="file" checked>
                                            <label class="btn btn-outline-primary" for="videoTypeFile" onclick="showFileUpload()">
                                                <i class="fas fa-upload me-2"></i>Upload File
                                            </label>
                                            
                                            <input type="radio" class="btn-check" name="video_type" id="videoTypeUrl" value="url">
                                            <label class="btn btn-outline-primary" for="videoTypeUrl" onclick="showUrlEmbed()">
                                                <i class="fas fa-link me-2"></i>Embed URL
                                            </label>
                                        </div>
                                        
                                        <script>
                                        function showFileUpload() {
                                            console.log('📁 Switching to file upload');
                                            document.getElementById('fileUploadSection').style.display = 'block';
                                            document.getElementById('urlEmbedSection').style.display = 'none';
                                            document.getElementById('videoTypeFile').checked = true;
                                            
                                            // Update requirements
                                            var videoFile = document.getElementById('videoFile');
                                            var videoUrl = document.getElementById('videoUrl');
                                            if (videoFile) videoFile.required = true;
                                            if (videoUrl) videoUrl.required = false;
                                        }
                                        
                                        function showUrlEmbed() {
                                            console.log('🔗 Switching to URL embed');
                                            document.getElementById('fileUploadSection').style.display = 'none';
                                            document.getElementById('urlEmbedSection').style.display = 'block';
                                            document.getElementById('videoTypeUrl').checked = true;
                                            
                                            // Update requirements
                                            var videoFile = document.getElementById('videoFile');
                                            var videoUrl = document.getElementById('videoUrl');
                                            if (videoFile) {
                                                videoFile.required = false;
                                                videoFile.value = '';
                                            }
                                            if (videoUrl) videoUrl.required = true;
                                            
                                            // Hide preview
                                            var preview = document.getElementById('previewContainer');
                                            if (preview) preview.style.display = 'none';
                                        }
                                        
                                        // Set initial state
                                        document.addEventListener('DOMContentLoaded', function() {
                                            showFileUpload();
                                        });
                                        </script>
                                    </div>

                                    <!-- Video File Upload -->
                                    <div class="mb-4" id="fileUploadSection">
                                        <label class="form-label fw-bold">Video File</label>
                                        <div class="upload-area" id="uploadArea">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" fill="currentColor" class="upload-icon mb-3" viewBox="0 0 16 16">
                                              <path d="M.5 9.9a.5.5 0 0 1 .5.5v2.5a1 1 0 0 0 1 1h12a1 1 0 0 0 1-1v-2.5a.5.5 0 0 1 1 0v2.5a2 2 0 0 1-2 2H2a2 2 0 0 1-2-2v-2.5a.5.5 0 0 1 .5-.5z"/>
                                              <path d="M7.646 1.146a.5.5 0 0 1 .708 0l3 3a.5.5 0 0 1-.708.708L8.5 2.707V11.5a.5.5 0 0 1-1 0V2.707L5.354 4.854a.5.5 0 1 1-.708-.708l3-3z"/>
                                            </svg>
                                            <h5 class="fw-bold mb-2">Drag & Drop your video here</h5>
                                            <p class="text-muted mb-3">or click to browse</p>
                                            <button type="button" class="btn btn-primary px-4 py-2 rounded-pill shadow-sm" id="browseBtn">
                                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-folder2-open me-2" viewBox="0 0 16 16">
                                                  <path d="M1 3.5A1.5 1.5 0 0 1 2.5 2h2.764c.958 0 1.76.56 2.311 1.184C7.985 3.648 8.48 4 9 4h4.5A1.5 1.5 0 0 1 15 5.5v.64c.57.265.94.876.856 1.546l-.64 5.124A2.5 2.5 0 0 1 12.733 15H3.266a2.5 2.5 0 0 1-2.481-2.19l-.64-5.124A1.5 1.5 0 0 1 1 6.14V3.5zM2 6h12v-.5a.5.5 0 0 0-.5-.5H9c-.964 0-1.71-.629-2.174-1.154C6.374 3.334 5.82 3 5.264 3H2.5a.5.5 0 0 0-.5.5V6zm-.367 1a.5.5 0 0 0-.496.562l.64 5.124A1.5 1.5 0 0 0 3.266 14h9.468a1.5 1.5 0 0 0 1.489-1.314l.64-5.124A.5.5 0 0 0 14.367 7H1.633z"/>
                                                </svg>
                                                Browse Files
                                            </button>
                                            <input type="file" id="videoFile" name="video" accept="video/*" style="display: none;">
                                        </div>
                                        
                                        <!-- Progress Bar (Initially Hidden) -->
                                        <div class="progress-container mt-4" id="progressContainer" style="display: none;">
                                            <h6 class="mb-2 fw-bold">Upload Progress</h6>
                                            <div class="progress mb-3" style="height: 10px;">
                                                <div class="progress-bar progress-bar-striped progress-bar-animated bg-primary" 
                                                     id="uploadProgress" role="progressbar" style="width: 0%">
                                                </div>
                                            </div>
                                            <div class="file-info p-3 border rounded bg-light" id="fileInfo">
                                                <div class="row">
                                                    <div class="col-md-6 mb-2">
                                                        <strong>File:</strong> <span id="fileName" class="text-truncate d-inline-block" style="max-width: 200px;"></span>
                                                    </div>
                                                    <div class="col-md-6 mb-2">
                                                        <strong>Size:</strong> <span id="fileSize"></span>
                                                    </div>
                                                </div>
                                                <div class="row">
                                                    <div class="col-md-6 mb-2">
                                                        <strong>Duration:</strong> <span id="videoDuration">Detecting...</span>
                                                    </div>
                                                    <div class="col-md-6 mb-2">
                                                        <strong>Status:</strong> <span id="uploadStatus" class="badge bg-info">Ready</span>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>

                                        <!-- Video Preview (Initially Hidden) -->
                                        <div class="preview-container mt-4" id="previewContainer" style="display: none;">
                                            <h6 class="mb-2 fw-bold">Video Preview</h6>
                                            <div class="video-preview-wrapper rounded overflow-hidden shadow-sm">
                                                <video id="videoPreview" class="video-preview w-100" controls></video>
                                            </div>
                                        </div>
                                    </div>

                                    <!-- Video URL Embedding -->
                                    <div class="mb-4" id="urlEmbedSection" style="display: none;">
                                        <label class="form-label fw-bold">Video URL</label>
                                        <div class="input-group">
                                            <span class="input-group-text">
                                                <i class="fas fa-video text-primary"></i>
                                            </span>
                                            <input type="url" class="form-control" id="videoUrl" name="video_url" placeholder="Paste video URL from YouTube, Vimeo, Facebook, TikTok, etc.">
                                            <button class="btn btn-outline-secondary" type="button" id="detectVideoBtn">
                                                <i class="fas fa-search me-1"></i>Detect
                                            </button>
                                        </div>
                                        <div class="form-text">
                                            <strong>Supported platforms:</strong> 
                                            <span class="badge bg-danger me-1">YouTube</span>
                                            <span class="badge bg-info me-1">Vimeo</span>
                                            <span class="badge bg-primary me-1">Facebook</span>
                                            <span class="badge bg-dark me-1">TikTok</span>
                                            <span class="badge bg-warning text-dark me-1">Dailymotion</span>
                                            <span class="badge bg-secondary">Wistia</span>
                                        </div>
                                        
                                        <!-- 🔍 DEBUG PANEL (Remove after fixing) -->
                                        <div id="debugPanel" class="mt-3 mb-3" style="background: #fff3cd; border: 1px solid #ffecb3; border-radius: 8px; padding: 15px;">
                                            <h6 style="color: #856404; margin-bottom: 10px;">
                                                <i class="fas fa-bug"></i> Debug Information (Live Status)
                                            </h6>
                                            <div id="debugOutput" style="font-family: monospace; font-size: 12px; background: white; padding: 10px; border-radius: 4px; max-height: 200px; overflow-y: auto;">
                                                <div style="color: #666;">🔄 Initializing debug panel...</div>
                                            </div>
                                            <button type="button" onclick="clearDebugPanel()" style="margin-top: 8px; padding: 5px 10px; background: #6c757d; color: white; border: none; border-radius: 3px; font-size: 11px;">
                                                Clear Debug
                                            </button>
                                            <button type="button" onclick="forceTestDetectionDirect()" style="margin-top: 8px; padding: 5px 10px; background: #28a745; color: white; border: none; border-radius: 3px; font-size: 11px;">
                                                Force Test Detection
                                            </button>
                                            <button type="button" onclick="checkJavaScript()" style="margin-top: 8px; padding: 5px 10px; background: #17a2b8; color: white; border: none; border-radius: 3px; font-size: 11px;">
                                                Check JS Status
                                            </button>
                                        </div>

                                        <!-- 🚀 BULLETPROOF DEBUG FUNCTIONS - Inline JavaScript -->
                                        <script>
                                            // Simple debug logging that always works
                                            function debugWrite(message, color = '#666') {
                                                const debugOutput = document.getElementById('debugOutput');
                                                if (debugOutput) {
                                                    const time = new Date().toLocaleTimeString();
                                                    debugOutput.innerHTML += `<div style="color: ${color}; margin-bottom: 2px;">[${time}] ${message}</div>`;
                                                    debugOutput.scrollTop = debugOutput.scrollHeight;
                                                } else {
                                                    console.log(message);
                                                }
                                            }

                                            // Clear debug panel
                                            function clearDebugPanel() {
                                                const debugOutput = document.getElementById('debugOutput');
                                                if (debugOutput) {
                                                    debugOutput.innerHTML = '<div style="color: #666;">Debug cleared...</div>';
                                                }
                                                debugWrite('🧹 Debug panel cleared', '#28a745');
                                            }

                                            // Check JavaScript status
                                            function checkJavaScript() {
                                                debugWrite('🔍 Checking JavaScript status...', '#17a2b8');
                                                
                                                // Check basic elements
                                                const videoUrl = document.getElementById('videoUrl');
                                                const detectBtn = document.getElementById('detectVideoBtn');
                                                
                                                debugWrite(`videoUrl element: ${videoUrl ? '✅ Found' : '❌ Missing'}`, videoUrl ? '#28a745' : '#dc3545');
                                                debugWrite(`detectBtn element: ${detectBtn ? '✅ Found' : '❌ Missing'}`, detectBtn ? '#28a745' : '#dc3545');
                                                debugWrite(`jQuery: ${typeof $ !== 'undefined' ? '✅ Loaded' : '❌ Missing'}`, typeof $ !== 'undefined' ? '#28a745' : '#dc3545');
                                                
                                                if (videoUrl) {
                                                    const currentValue = videoUrl.value;
                                                    debugWrite(`Current URL: "${currentValue}"`, '#6c757d');
                                                    
                                                    if (currentValue) {
                                                        debugWrite('🎯 URL found - you can test detection now!', '#28a745');
                                                    } else {
                                                        debugWrite('⚠️ No URL entered - paste a YouTube URL first', '#ffc107');
                                                    }
                                                }
                                            }

                                            // Force test detection with pure JavaScript
                                            function forceTestDetectionDirect() {
                                                debugWrite('🚀 Starting manual detection test...', '#17a2b8');
                                                
                                                const videoUrl = document.getElementById('videoUrl');
                                                if (!videoUrl) {
                                                    debugWrite('❌ Video URL input not found!', '#dc3545');
                                                    return;
                                                }
                                                
                                                const url = videoUrl.value.trim() || 'https://youtu.be/dCLhUIakPQ3s';
                                                if (!videoUrl.value.trim()) {
                                                    videoUrl.value = url;
                                                    debugWrite(`🔗 Using test URL: ${url}`, '#6c757d');
                                                }
                                                
                                                debugWrite('📡 Sending AJAX request to detect_video.php...', '#17a2b8');
                                                
                                                // Use pure JavaScript XMLHttpRequest (no jQuery dependency)
                                                const xhr = new XMLHttpRequest();
                                                xhr.open('POST', 'ajax/detect_video.php', true);
                                                xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
                                                
                                                xhr.onload = function() {
                                                    if (xhr.status === 200) {
                                                        debugWrite('✅ AJAX response received!', '#28a745');
                                                        debugWrite(`Response: ${xhr.responseText}`, '#6c757d');
                                                        
                                                        try {
                                                            const response = JSON.parse(xhr.responseText);
                                                            debugWrite('✅ JSON parsed successfully', '#28a745');
                                                            
                                                            if (response.success && response.data && response.data.is_valid) {
                                                                debugWrite(`🎉 Video detected: ${response.data.platform_name}`, '#28a745');
                                                                debugWrite(`Video ID: ${response.data.video_id}`, '#6c757d');
                                                                
                                                                // Show success message
                                                                alert(`✅ Success! ${response.data.platform_name} video detected!\nVideo ID: ${response.data.video_id}`);
                                                            } else {
                                                                debugWrite('❌ Detection failed - invalid response', '#dc3545');
                                                                debugWrite(`Error: ${response.message || 'Unknown error'}`, '#dc3545');
                                                            }
                                                        } catch (e) {
                                                            debugWrite(`❌ JSON parse error: ${e.message}`, '#dc3545');
                                                        }
                                                    } else {
                                                        debugWrite(`❌ HTTP Error: ${xhr.status} ${xhr.statusText}`, '#dc3545');
                                                    }
                                                };
                                                
                                                xhr.onerror = function() {
                                                    debugWrite('❌ Network error occurred', '#dc3545');
                                                };
                                                
                                                xhr.send('video_url=' + encodeURIComponent(url));
                                            }

                                            // Add auto-detection on URL input
                                            function setupAutoDetection() {
                                                const videoUrlInput = document.getElementById('videoUrl');
                                                if (videoUrlInput) {
                                                    debugWrite('🔗 Setting up auto-detection on URL input', '#17a2b8');
                                                    
                                                    let autoDetectTimeout;
                                                    
                                                    // Use native addEventListener for better compatibility
                                                    ['input', 'paste', 'keyup'].forEach(eventType => {
                                                        videoUrlInput.addEventListener(eventType, function() {
                                                            clearTimeout(autoDetectTimeout);
                                                            
                                                            const url = this.value.trim();
                                                            debugWrite(`🎥 URL input detected (${eventType}): "${url.substring(0, 50)}${url.length > 50 ? '...' : ''}"`, '#6c757d');
                                                            
                                                            if (url.length > 10) {
                                                                // Check if it looks like a video URL
                                                                const isVideoUrl = /(?:youtube\.com|youtu\.be|vimeo\.com|facebook\.com|tiktok\.com|dailymotion\.com|wistia\.com)/i.test(url);
                                                                
                                                                if (isVideoUrl) {
                                                                    debugWrite('✅ Valid video URL detected - auto-detection starting in 1 second...', '#28a745');
                                                                    
                                                                    // Auto-detect after 1 second delay
                                                                    autoDetectTimeout = setTimeout(function() {
                                                                        debugWrite('🚀 Auto-detection triggered!', '#17a2b8');
                                                                        forceTestDetectionDirect();
                                                                    }, 1000);
                                                                } else {
                                                                    debugWrite('⚠️ URL does not match supported platforms', '#ffc107');
                                                                }
                                                            }
                                                        });
                                                    });
                                                    
                                                    debugWrite('✅ Auto-detection listeners attached successfully!', '#28a745');
                                                } else {
                                                    debugWrite('❌ Cannot setup auto-detection - videoUrl element not found', '#dc3545');
                                                }
                                            }

                                            // Initialize debug system immediately
                                            setTimeout(function() {
                                                debugWrite('🚀 Bulletproof debug system loaded!', '#28a745');
                                                checkJavaScript();
                                                setupAutoDetection();
                                            }, 100);
                                        </script>

                                        <!-- Dynamic Video Detection Display -->
                                        <div id="videoDetectionResult" class="mt-3" style="display: none;">
                                            <div class="card border-success">
                                                <div class="card-header bg-success text-white">
                                                    <h6 class="card-title mb-0">
                                                        <i class="fas fa-check-circle me-2"></i>
                                                        <span id="detectedPlatform"></span> Video Confirmed
                                                    </h6>
                                                </div>
                                                <div class="card-body">
                                                    <div class="row">
                                                        <div class="col-md-6">
                                                            <h6 class="fw-bold mb-3">📋 Extracted Data:</h6>
                                                            <table class="table table-sm table-borderless">
                                                                <tr>
                                                                    <td class="fw-bold text-muted">Platform:</td>
                                                                    <td><span id="platformBadge" class="badge bg-primary"></span></td>
                                                                </tr>
                                                                <tr>
                                                                    <td class="fw-bold text-muted">Video ID:</td>
                                                                    <td><code id="extractedVideoId" class="small"></code></td>
                                                                </tr>
                                                                <tr>
                                                                    <td class="fw-bold text-muted">Original URL:</td>
                                                                    <td><span id="detectedUrl" class="small text-break"></span></td>
                                                                </tr>
                                                                <tr>
                                                                    <td class="fw-bold text-muted">Thumbnail:</td>
                                                                    <td><span id="thumbnailStatus" class="small"></span></td>
                                                                </tr>
                                                            </table>
                                                            
                                                            <div class="mt-3">
                                                                <button type="button" class="btn btn-sm btn-outline-primary" id="previewEmbedBtn">
                                                                    <i class="fas fa-eye me-1"></i>Preview Embed
                                                                </button>
                                                                <button type="button" class="btn btn-sm btn-outline-info ms-2" id="showRawDataBtn">
                                                                    <i class="fas fa-code me-1"></i>Raw Data
                                                                </button>
                                                            </div>
                                                        </div>
                                                        <div class="col-md-6">
                                                            <h6 class="fw-bold mb-3">🖼️ Thumbnail Preview:</h6>
                                                            <div id="thumbnailPreviewContainer" class="text-center">
                                                                <img id="thumbnailPreview" class="img-fluid rounded shadow-sm" style="max-height: 150px; display: none;" alt="Video Thumbnail">
                                                                <div id="thumbnailPlaceholder" class="bg-light border rounded d-flex align-items-center justify-content-center" style="height: 150px;">
                                                                    <span class="text-muted">No thumbnail available</span>
                                                                </div>
                                                            </div>
                                                            
                                                            <!-- Real-time sync status -->
                                                            <div class="mt-3">
                                                                <div class="d-flex align-items-center">
                                                                    <div class="spinner-border spinner-border-sm text-success me-2" role="status" id="syncSpinner" style="display: none;"></div>
                                                                    <small id="syncStatus" class="text-success fw-bold">✅ Sync Complete</small>
                                                                </div>
                                                                <small class="text-muted d-block" id="syncTime"></small>
                                                            </div>
                                                        </div>
                                                    </div>
                                                    
                                                    <!-- Raw Data Display (Hidden by default) -->
                                                    <div id="rawDataDisplay" class="mt-4" style="display: none;">
                                                        <h6 class="fw-bold">🔍 Complete Response Data:</h6>
                                                        <pre id="rawDataContent" class="bg-light p-3 rounded small" style="max-height: 200px; overflow-y: auto;"></pre>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <!-- Embed Preview -->
                                        <div id="embedPreviewContainer" class="mt-3" style="display: none;">
                                            <h6 class="mb-2 fw-bold">Embed Preview</h6>
                                            <div class="embed-preview-wrapper rounded overflow-hidden shadow-sm bg-light p-3">
                                                <div id="embedPreview"></div>
                                            </div>
                                        </div>
                                        
                                        <!-- Hidden fields for embed data -->
                                        <input type="hidden" id="detectedPlatformInput" name="detected_platform">
                                        <input type="hidden" id="videoIdExternal" name="video_id_external">
                                        <input type="hidden" id="embedCodeInput" name="embed_code">
                                    </div>

                                    <!-- Video Details -->
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label for="videoTitle" class="form-label">Video Title <span class="text-danger">*</span></label>
                                                <input type="text" class="form-control" id="videoTitle" name="title" required>
                                                <div class="invalid-feedback">
                                                    Please provide a video title.
                                                </div>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label for="courseSelect" class="form-label">Course <span class="text-danger">*</span></label>
                                                <select class="form-select" id="courseSelect" name="course_id" required>
                                                    <option value="">Select a course</option>
                                                    <?php foreach ($courses as $course): ?>
                                                        <option value="<?php echo $course['course_id']; ?>">
                                                            <?php echo htmlspecialchars($course['title']); ?>
                                                        </option>
                                                    <?php endforeach; ?>
                                                </select>
                                                <div class="invalid-feedback">
                                                    Please select a course for this video.
                                                </div>
                                            </div>
                                        </div>
                                    </div>

                                    <div class="mb-3">
                                        <label for="videoDescription" class="form-label">Description</label>
                                        <textarea class="form-control" id="videoDescription" name="description" rows="4"></textarea>
                                    </div>

                                    <!-- Thumbnail Upload -->
                                    <div class="mb-3">
                                        <label for="thumbnailFile" class="form-label">Thumbnail (Optional)</label>
                                        <input type="file" class="form-control" id="thumbnailFile" name="thumbnail" accept="image/*">
                                        <div class="form-text">Upload a custom thumbnail image (recommended: 1280x720)</div>
                                        <div id="thumbnailPreview" class="mt-2" style="display: none;">
                                            <img id="thumbnailImg" class="thumbnail-preview" alt="Thumbnail preview">
                                        </div>
                                    </div>

                                    <!-- Additional Settings -->
                                    <div class="row">
                                        <div class="col-md-6">
                                                <div class="mb-3">
                                                <label for="moduleOrder" class="form-label">Module Order</label>
                                                <input type="number" class="form-control" id="moduleOrder" name="module_order" min="1" value="1">
                                                <div class="invalid-feedback">
                                                    Module order must be a positive number.
                                                </div>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label for="videoPrice" class="form-label">Price (R)</label>
                                                <input type="number" class="form-control" id="videoPrice" name="price" min="0" step="0.01" value="0">
                                                <div class="invalid-feedback">
                                                    Price must be zero or greater.
                                                </div>
                                            </div>
                                        </div>
                                    </div>

                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label for="categoryName" class="form-label">Category</label>
                                                <input type="text" class="form-control" id="categoryName" name="category_name" placeholder="e.g., Programming, Design">
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label class="form-label">Visibility</label>
                                                <div class="form-check">
                                                    <input class="form-check-input" type="radio" name="is_published" id="published" value="1" checked>
                                                    <label class="form-check-label" for="published">Published</label>
                                                </div>
                                                <div class="form-check">
                                                    <input class="form-check-input" type="radio" name="is_published" id="draft" value="0">
                                                    <label class="form-check-label" for="draft">Draft</label>
                                                </div>
                                            </div>
                                        </div>
                                    </div>

                                    <div class="mb-3">
                                        <div class="form-check">
                                            <input class="form-check-input" type="checkbox" id="isFree" name="is_free" value="1">
                                            <label class="form-check-label" for="isFree">
                                                This video is free to watch
                                            </label>
                                        </div>
                                    </div>

                                    <!-- Video-Specific Content Creation -->
                                    <div class="mb-4">
                                        <label class="form-label fw-bold mb-3">
                                            <i class="fas fa-plus-circle me-2 text-primary"></i>Additional Learning Materials
                                        </label>
                                        <div class="card border-0 bg-light">
                                            <div class="card-body">
                                                <p class="text-muted mb-3 small">
                                                    <i class="fas fa-info-circle me-1"></i>
                                                    Select the types of learning materials you want to create for this video. After uploading, you'll be guided through creating each selected item.
                                                </p>
                                                
                                                <div class="row">
                                                    <div class="col-md-6 mb-3">
                                                        <div class="form-check form-check-card">
                                                            <input class="form-check-input" type="checkbox" id="hasQuiz" name="content_types[]" value="quiz">
                                                            <label class="form-check-label content-type-label" for="hasQuiz">
                                                                <div class="d-flex align-items-center">
                                                                    <div class="content-icon bg-warning bg-opacity-10 text-warning me-3">
                                                                        <i class="fas fa-brain"></i>
                                                                    </div>
                                                                    <div>
                                                                        <div class="fw-bold">Interactive Quiz</div>
                                                                        <small class="text-muted">Test student understanding</small>
                                                                    </div>
                                                                </div>
                                                            </label>
                                                        </div>
                                                    </div>
                                                    
                                                    <div class="col-md-6 mb-3">
                                                        <div class="form-check form-check-card">
                                                            <input class="form-check-input" type="checkbox" id="hasAssignment" name="content_types[]" value="assignment">
                                                            <label class="form-check-label content-type-label" for="hasAssignment">
                                                                <div class="d-flex align-items-center">
                                                                    <div class="content-icon bg-primary bg-opacity-10 text-primary me-3">
                                                                        <i class="fas fa-edit"></i>
                                                                    </div>
                                                                    <div>
                                                                        <div class="fw-bold">Practice Assignment</div>
                                                                        <small class="text-muted">Hands-on exercise</small>
                                                                    </div>
                                                                </div>
                                                            </label>
                                                        </div>
                                                    </div>
                                                    
                                                    <div class="col-md-6 mb-3">
                                                        <div class="form-check form-check-card">
                                                            <input class="form-check-input" type="checkbox" id="hasMaterials" name="content_types[]" value="materials">
                                                            <label class="form-check-label content-type-label" for="hasMaterials">
                                                                <div class="d-flex align-items-center">
                                                                    <div class="content-icon bg-success bg-opacity-10 text-success me-3">
                                                                        <i class="fas fa-file-alt"></i>
                                                                    </div>
                                                                    <div>
                                                                        <div class="fw-bold">Supplementary Resources</div>
                                                                        <small class="text-muted">Downloads, PDFs, links</small>
                                                                    </div>
                                                                </div>
                                                            </label>
                                                        </div>
                                                    </div>
                                                    
                                                    <div class="col-md-6 mb-3">
                                                        <div class="form-check form-check-card">
                                                            <input class="form-check-input" type="checkbox" id="hasNotes" name="content_types[]" value="notes">
                                                            <label class="form-check-label content-type-label" for="hasNotes">
                                                                <div class="d-flex align-items-center">
                                                                    <div class="content-icon bg-info bg-opacity-10 text-info me-3">
                                                                        <i class="fas fa-sticky-note"></i>
                                                                    </div>
                                                                    <div>
                                                                        <div class="fw-bold">Lecture Notes</div>
                                                                        <small class="text-muted">Study guide, transcript</small>
                                                                    </div>
                                                                </div>
                                                            </label>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>

                                    <!-- Form Actions -->
                                    <div class="d-flex justify-content-between mt-4">
                                        <button type="button" class="btn btn-outline-secondary rounded-pill px-4" id="resetBtn">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-arrow-counterclockwise me-2" viewBox="0 0 16 16">
                                              <path fill-rule="evenodd" d="M8 3a5 5 0 1 1-4.546 2.914.5.5 0 0 0-.908-.417A6 6 0 1 0 8 2v1z"/>
                                              <path d="M8 4.466V.534a.25.25 0 0 0-.41-.192L5.23 2.308a.25.25 0 0 0 0 .384l2.36 1.966A.25.25 0 0 0 8 4.466z"/>
                                            </svg> Reset Form
                                        </button>
                                        <button type="submit" class="btn btn-primary rounded-pill px-4 py-2 shadow-sm" id="submitBtn">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-cloud-arrow-up me-2" viewBox="0 0 16 16">
                                              <path fill-rule="evenodd" d="M7.646 5.146a.5.5 0 0 1 .708 0l2 2a.5.5 0 0 1-.708.708L8.5 6.707V10.5a.5.5 0 0 1-1 0V6.707L6.354 7.854a.5.5 0 1 1-.708-.708l2-2z"/>
                                              <path d="M4.406 3.342A5.53 5.53 0 0 1 8 2c2.69 0 4.923 2 5.166 4.579C14.758 6.804 16 8.137 16 9.773 16 11.569 14.502 13 12.687 13H3.781C1.708 13 0 11.366 0 9.318c0-1.763 1.266-3.223 2.942-3.593.143-.863.698-1.723 1.464-2.383z"/>
                                            </svg> Upload Video
                                        </button>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>

                    <!-- Upload Tips -->
                    <div class="col-lg-4">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="card-title mb-0">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" fill="#ffc107" class="bi bi-lightbulb me-2" viewBox="0 0 16 16">
                                      <path d="M8 16A8 8 0 1 1 8 0a8 8 0 0 1 0 16zM2.5 8a5.5 5.5 0 1 1 11 0 5.5 5.5 0 0 1-11 0z"/>
                                      <path d="M9.5 6.5a1.5 1.5 0 0 0-1-1.415V2.5A.5.5 0 0 0 8 2a.5.5 0 0 0-.5.5v2.585a1.5 1.5 0 0 0 0 2.83V12h1V7.915a1.5 1.5 0 0 0 0-1.415z"/>
                                    </svg>
                                    Upload Tips
                                </h5>
                            </div>
                            <div class="card-body p-4">
                                <ul class="list-unstyled upload-tips-list">
                                    <li class="mb-3 d-flex align-items-center">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" fill="#198754" class="bi bi-check-circle-fill me-2" viewBox="0 0 16 16">
                                          <path d="M16 8A8 8 0 1 1 0 8a8 8 0 0 1 16 0zm-3.97-3.03a.75.75 0 0 0-1.08.022L7.477 9.417 5.384 7.323a.75.75 0 0 0-1.06 1.06L6.97 11.03a.75.75 0 0 0 1.079-.02l3.992-4.99a.75.75 0 0 0-.01-1.05z"/>
                                        </svg>
                                        <span>Supported formats: MP4, WebM, MOV, AVI</span>
                                    </li>
                                    <li class="mb-3 d-flex align-items-center">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" fill="#198754" class="bi bi-check-circle-fill me-2" viewBox="0 0 16 16">
                                          <path d="M16 8A8 8 0 1 1 0 8a8 8 0 0 1 16 0zm-3.97-3.03a.75.75 0 0 0-1.08.022L7.477 9.417 5.384 7.323a.75.75 0 0 0-1.06 1.06L6.97 11.03a.75.75 0 0 0 1.079-.02l3.992-4.99a.75.75 0 0 0-.01-1.05z"/>
                                        </svg>
                                        <span>No file size limit</span>
                                    </li>
                                    <li class="mb-3 d-flex align-items-center">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" fill="#198754" class="bi bi-check-circle-fill me-2" viewBox="0 0 16 16">
                                          <path d="M16 8A8 8 0 1 1 0 8a8 8 0 0 1 16 0zm-3.97-3.03a.75.75 0 0 0-1.08.022L7.477 9.417 5.384 7.323a.75.75 0 0 0-1.06 1.06L6.97 11.03a.75.75 0 0 0 1.079-.02l3.992-4.99a.75.75 0 0 0-.01-1.05z"/>
                                        </svg>
                                        <span>Recommended resolution: 1080p or higher</span>
                                    </li>
                                    <li class="mb-3 d-flex align-items-center">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" fill="#198754" class="bi bi-check-circle-fill me-2" viewBox="0 0 16 16">
                                          <path d="M16 8A8 8 0 1 1 0 8a8 8 0 0 1 16 0zm-3.97-3.03a.75.75 0 0 0-1.08.022L7.477 9.417 5.384 7.323a.75.75 0 0 0-1.06 1.06L6.97 11.03a.75.75 0 0 0 1.079-.02l3.992-4.99a.75.75 0 0 0-.01-1.05z"/>
                                        </svg>
                                        <span>Custom thumbnails improve engagement</span>
                                    </li>
                                    <li class="mb-3 d-flex align-items-center">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" fill="#198754" class="bi bi-check-circle-fill me-2" viewBox="0 0 16 16">
                                          <path d="M16 8A8 8 0 1 1 0 8a8 8 0 0 1 16 0zm-3.97-3.03a.75.75 0 0 0-1.08.022L7.477 9.417 5.384 7.323a.75.75 0 0 0-1.06 1.06L6.97 11.03a.75.75 0 0 0 1.079-.02l3.992-4.99a.75.75 0 0 0-.01-1.05z"/>
                                        </svg>
                                        <span>Clear titles help with discovery</span>
                                    </li>
                                </ul>
                            </div>
                        </div>

                        <div class="card mt-3">
                            <div class="card-header">
                                <h6 class="card-title mb-0">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="#0dcaf0" class="bi bi-info-circle me-2" viewBox="0 0 16 16">
                                      <path d="M8 15A7 7 0 1 1 8 1a7 7 0 0 1 0 14zm0 1A8 8 0 1 0 8 0a8 8 0 0 0 0 16z"/>
                                      <path d="m8.93 6.588-2.29.287-.082.38.45.083c.294.07.352.176.288.469l-.738 3.468c-.194.897.105 1.319.808 1.319.545 0 1.178-.252 1.465-.598l.088-.416c-.2.176-.492.246-.686.246-.275 0-.375-.193-.304-.533L8.93 6.588zM9 4.5a1 1 0 1 1-2 0 1 1 0 0 1 2 0z"/>
                                    </svg>
                                    Upload Status
                                </h6>
                            </div>
                            <div class="card-body p-4">
                                <div id="uploadStatusCard" class="p-3 rounded-3 bg-light border">
                                    <div class="d-flex align-items-center">
                                        <span class="badge bg-secondary me-2 px-3 py-2">Idle</span>
                                        <p class="text-muted mb-0 ms-2">No upload in progress</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <!-- Custom CSS for Content Type Selection -->
    <style>
        /* Content Type Selection Styling */
        .form-check-card {
            border: 2px solid #e9ecef;
            border-radius: 12px;
            padding: 0;
            transition: all 0.2s ease;
            background: white;
        }

        .form-check-card:hover {
            border-color: #0d6efd;
            box-shadow: 0 2px 8px rgba(13, 110, 253, 0.15);
        }

        .form-check-card input[type="checkbox"]:checked + .content-type-label {
            background: rgba(13, 110, 253, 0.05);
            border-color: #0d6efd;
        }

        .content-type-label {
            display: block;
            padding: 15px;
            margin: 0;
            cursor: pointer;
            border-radius: 10px;
            transition: all 0.2s ease;
            width: 100%;
        }

        .content-type-label:hover {
            background: rgba(13, 110, 253, 0.02);
        }

        .content-icon {
            width: 40px;
            height: 40px;
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.1rem;
        }

        .form-check-input {
            position: absolute;
            top: 15px;
            right: 15px;
            width: 20px;
            height: 20px;
            margin: 0;
        }

        /* Selected state animations */
        .form-check-card input[type="checkbox"]:checked + .content-type-label {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(13, 110, 253, 0.2);
        }

        /* Content type specific colors */
        .form-check-card input[type="checkbox"]:checked + .content-type-label .content-icon.bg-warning {
            background: #ffc107 !important;
            color: white !important;
        }

        .form-check-card input[type="checkbox"]:checked + .content-type-label .content-icon.bg-primary {
            background: #0d6efd !important;
            color: white !important;
        }

        .form-check-card input[type="checkbox"]:checked + .content-type-label .content-icon.bg-success {
            background: #198754 !important;
            color: white !important;
        }

        .form-check-card input[type="checkbox"]:checked + .content-type-label .content-icon.bg-info {
            background: #0dcaf0 !important;
            color: white !important;
        }
    </style>

    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script type="text/javascript" src="https://cdn.jsdelivr.net/npm/toastify-js"></script>
    <script src="video-upload.js"></script>
    <script>
        // Sidebar toggle functionality
        document.getElementById('sidebarToggle').addEventListener('click', function() {
            document.querySelector('.sidebar').classList.toggle('show');
        });
        
        // Alert helper function
        function showAlert(message, type = 'info') {
            const alertContainer = $('#alertContainer');
            const alertHtml = `
                <div class="alert alert-${type} alert-dismissible fade show" role="alert">
                    <i class="fas fa-${type === 'success' ? 'check-circle' : type === 'danger' ? 'exclamation-triangle' : type === 'warning' ? 'exclamation-circle' : 'info-circle'} me-2"></i>
                    ${message}
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            `;
            
            // Create alert container if it doesn't exist
            if (alertContainer.length === 0) {
                $('form').prepend('<div id="alertContainer"></div>');
            }
            
            $('#alertContainer').html(alertHtml);
            
            // Auto-hide success and info alerts after 5 seconds
            if (type === 'success' || type === 'info') {
                setTimeout(() => {
                    $('.alert').fadeOut();
                }, 5000);
            }
        }

        // 📝 DEBUG LOGGING FUNCTION - Writes to visual debug panel
        function debugLog(message, type = 'info') {
            const debugOutput = document.getElementById('debugOutput');
            if (debugOutput) {
                const timestamp = new Date().toLocaleTimeString();
                const color = type === 'error' ? '#dc3545' : type === 'success' ? '#28a745' : type === 'warning' ? '#ffc107' : '#6c757d';
                const icon = type === 'error' ? '❌' : type === 'success' ? '✅' : type === 'warning' ? '⚠️' : 'ℹ️';
                
                const logEntry = document.createElement('div');
                logEntry.style.color = color;
                logEntry.style.marginBottom = '2px';
                logEntry.innerHTML = `<span style="color: #666;">[${timestamp}]</span> ${icon} ${message}`;
                
                debugOutput.appendChild(logEntry);
                debugOutput.scrollTop = debugOutput.scrollHeight;
            }
            
            // Also log to console
            console.log(message);
        }

        // 🔍 COMPREHENSIVE DEBUGGING - Visual status check
        function runElementCheck() {
            debugLog('🔍 Starting element existence check...', 'info');
            
            const videoUrlElement = document.getElementById('videoUrl');
            const detectBtnElement = document.getElementById('detectVideoBtn');
            const resultDivElement = document.getElementById('videoDetectionResult');
            
            debugLog(`videoUrl element: ${videoUrlElement ? '✅ Found' : '❌ Not found'}`, videoUrlElement ? 'success' : 'error');
            debugLog(`detectVideoBtn element: ${detectBtnElement ? '✅ Found' : '❌ Not found'}`, detectBtnElement ? 'success' : 'error');
            debugLog(`videoDetectionResult element: ${resultDivElement ? '✅ Found' : '❌ Not found'}`, resultDivElement ? 'success' : 'error');
            debugLog(`jQuery version: ${$.fn.jquery || 'Not loaded'}`, $.fn.jquery ? 'success' : 'error');
            debugLog(`Bootstrap: ${typeof bootstrap !== 'undefined' ? '✅ Loaded' : '❌ Not loaded'}`, typeof bootstrap !== 'undefined' ? 'success' : 'error');
            
            if (videoUrlElement) {
                const currentValue = videoUrlElement.value;
                debugLog(`Current URL value: "${currentValue}"`, currentValue ? 'info' : 'warning');
            }
        }
        
        // 🎯 TEST DETECTION FUNCTION - Forces manual detection test
        function forceTestDetection() {
            debugLog('🚀 Forcing detection test...', 'info');
            
            const videoUrl = document.getElementById('videoUrl');
            if (!videoUrl) {
                debugLog('❌ Cannot test - videoUrl element not found', 'error');
                return;
            }
            
            const testUrl = videoUrl.value || 'https://youtu.be/dCLhUIakPQ3s';
            videoUrl.value = testUrl;
            debugLog(`Using test URL: ${testUrl}`, 'info');
            
            // Trigger the detection manually
            debugLog('📡 Sending AJAX request...', 'info');
            
            $.ajax({
                url: 'ajax/detect_video.php',
                method: 'POST',
                data: { video_url: testUrl },
                dataType: 'json',
                success: function(response) {
                    debugLog('🎉 AJAX Success! Response received', 'success');
                    debugLog(`Response: ${JSON.stringify(response)}`, 'info');
                    
                    if (response.success && response.data && response.data.is_valid) {
                        debugLog(`✅ Video detected: ${response.data.platform_name}`, 'success');
                        debugLog(`Video ID: ${response.data.video_id}`, 'info');
                        showAlert(`✅ Detection test successful! ${response.data.platform_name} video found.`, 'success');
                    } else {
                        debugLog('❌ Detection failed - invalid response', 'error');
                        showAlert('❌ Detection test failed - check debug panel', 'danger');
                    }
                },
                error: function(xhr, status, error) {
                    debugLog(`❌ AJAX Error: ${error}`, 'error');
                    debugLog(`Status: ${status}`, 'error');
                    debugLog(`Response: ${xhr.responseText}`, 'error');
                    showAlert(`❌ AJAX Error: ${error}`, 'danger');
                }
            });
        }
        
        // Initialize debugging immediately when script loads
        debugLog('🚀 Debug system initialized', 'success');
        
        // Run initial element check
        setTimeout(function() {
            runElementCheck();
            debugLog('🎯 Ready for video detection testing', 'info');
        }, 1000);

        // Form validation and handling
        $(document).ready(function() {
            debugLog('🎥 Video Upload Form Initialized - jQuery ready', 'success');
            
            // 🔗 Connect debug buttons
            $('#clearDebugBtn').click(function() {
                const debugOutput = document.getElementById('debugOutput');
                if (debugOutput) {
                    debugOutput.innerHTML = '<div style="color: #666;">Debug cleared...</div>';
                }
            });
            
            $('#testDetectionBtn').click(function() {
                forceTestDetection();
            });
            
            // Wait a moment for DOM to be fully ready
            setTimeout(function() {
                // Bulletproof form switching
                function switchVideoMode(mode) {
                    console.log('🔄 Switching to mode:', mode);
                    
                    try {
                        if (mode === 'url') {
                            // Show URL section, hide file section
                            if ($('#urlEmbedSection').length) $('#urlEmbedSection').show();
                            if ($('#fileUploadSection').length) $('#fileUploadSection').hide();
                            
                            // Update form requirements
                            if ($('#videoUrl').length) $('#videoUrl').prop('required', true);
                            if ($('#videoFile').length) $('#videoFile').prop('required', false);
                            
                            // Clear file input
                            if ($('#videoFile').length) $('#videoFile').val('');
                            if ($('#previewContainer').length) $('#previewContainer').hide();
                            
                            console.log('✅ URL mode activated');
                        } else {
                            // Show file section, hide URL section  
                            if ($('#fileUploadSection').length) $('#fileUploadSection').show();
                            if ($('#urlEmbedSection').length) $('#urlEmbedSection').hide();
                            
                            // Update form requirements
                            if ($('#videoFile').length) $('#videoFile').prop('required', true);
                            if ($('#videoUrl').length) $('#videoUrl').prop('required', false);
                            
                            console.log('✅ File mode activated');
                        }
                    } catch (error) {
                        console.error('❌ Error switching video mode:', error);
                    }
                }
                
                // Set initial state
                switchVideoMode('file');
                
                // Handle Bootstrap button clicks with maximum compatibility
                $(document).off('click.videoswitch').on('click.videoswitch', '.btn-group label', function(e) {
                    const buttonFor = $(this).attr('for');
                    console.log('🖱️ Button clicked:', buttonFor);
                    
                    if (buttonFor === 'videoTypeUrl') {
                        switchVideoMode('url');
                    } else if (buttonFor === 'videoTypeFile') {
                        switchVideoMode('file');
                    }
                });
                
                // Handle radio button changes as backup
                $(document).off('change.videoswitch').on('change.videoswitch', 'input[name="video_type"]', function() {
                    const value = $(this).val();
                    console.log('📻 Radio changed to:', value);
                    switchVideoMode(value);
                });
                
                // Handle direct clicks on radio inputs
                $(document).off('click.videoswitch').on('click.videoswitch', '#videoTypeFile, #videoTypeUrl', function() {
                    const value = $(this).val();
                    console.log('🔘 Direct radio click:', value);
                    switchVideoMode(value);
                });
                
            }, 100); // Small delay to ensure DOM is ready

            // Video URL detection with dynamic sync confirmation
            $('#detectVideoBtn').click(function() {
                const videoUrl = $('#videoUrl').val().trim();
                
                if (!videoUrl) {
                    showAlert('Please enter a video URL first.', 'warning');
                    return;
                }
                
                // Start sync process with visual feedback
                startSyncProcess();
                
                // Show loading state
                $(this).html('<i class="fas fa-spinner fa-spin me-1"></i>Detecting...');
                $(this).prop('disabled', true);
                
                // Hide previous results
                $('#videoDetectionResult').hide();
                
                // Detect video using AJAX
                $.ajax({
                    url: 'ajax/detect_video.php',
                    method: 'POST',
                    data: { video_url: videoUrl },
                    dataType: 'json',
                    success: function(response) {
                        console.log('🎥 Detection response:', response);
                        
                        if (response.success && response.data.is_valid) {
                            const videoData = response.data;
                            
                            // Populate dynamic sync confirmation display
                            populateDetectionDisplay(videoData);
                            
                            // Store hidden field values for form submission
                            $('#detectedPlatformInput').val(videoData.platform);
                            $('#videoIdExternal').val(videoData.video_id);
                            $('#embedCodeInput').val(videoData.embed_code);
                            
                            // Auto-fill title if empty
                            if (!$('#videoTitle').val() && videoData.suggested_title) {
                                $('#videoTitle').val(videoData.suggested_title);
                            }
                            
                            // Complete sync process successfully
                            completeSyncProcess(true);
                            
                            showAlert('🎉 Video detected and synced successfully from ' + videoData.platform_name + '!', 'success');
                        } else {
                            completeSyncProcess(false, response.message || 'Detection failed');
                            showAlert(response.message || 'Unable to detect video. Please check the URL and try again.', 'danger');
                            $('#videoDetectionResult').hide();
                        }
                    },
                    error: function(xhr, status, error) {
                        console.error('🚨 Detection error:', error);
                        completeSyncProcess(false, 'Network error: ' + error);
                        showAlert('Error detecting video. Please check your connection and try again.', 'danger');
                        $('#videoDetectionResult').hide();
                    },
                    complete: function() {
                        // Reset button state
                        $('#detectVideoBtn').html('<i class="fas fa-search me-1"></i>Detect');
                        $('#detectVideoBtn').prop('disabled', false);
                    }
                });
            });

            // Start sync process with visual feedback
            function startSyncProcess() {
                const syncSpinner = $('#syncSpinner');
                const syncStatus = $('#syncStatus');
                const syncTime = $('#syncTime');
                
                // Show spinner and update status
                syncSpinner.show();
                syncStatus.text('🔄 Syncing with platform...').removeClass().addClass('text-primary fw-bold');
                syncTime.text('Started: ' + new Date().toLocaleTimeString());
            }
            
            // Complete sync process with success/error feedback
            function completeSyncProcess(success, errorMessage = null) {
                const syncSpinner = $('#syncSpinner');
                const syncStatus = $('#syncStatus');
                const syncTime = $('#syncTime');
                
                // Hide spinner
                syncSpinner.hide();
                
                if (success) {
                    syncStatus.text('✅ Sync Complete').removeClass().addClass('text-success fw-bold');
                    syncTime.text('Completed: ' + new Date().toLocaleTimeString());
                } else {
                    syncStatus.text('❌ Sync Failed').removeClass().addClass('text-danger fw-bold');
                    syncTime.text('Failed: ' + new Date().toLocaleTimeString() + 
                                  (errorMessage ? ' - ' + errorMessage : ''));
                }
            }
            
            // Populate the detection display with comprehensive response data
            function populateDetectionDisplay(videoData) {
                // Platform information
                $('#detectedPlatform').text(videoData.platform_name);
                $('#platformBadge').text(videoData.platform_name)
                    .removeClass().addClass('badge bg-' + getPlatformColor(videoData.platform));
                
                // Video details
                $('#extractedVideoId').text(videoData.video_id);
                $('#detectedUrl').text(videoData.original_url);
                
                // Thumbnail handling
                if (videoData.thumbnail_url) {
                    const thumbnailImg = $('#thumbnailPreview');
                    const thumbnailPlaceholder = $('#thumbnailPlaceholder');
                    const thumbnailStatus = $('#thumbnailStatus');
                    
                    thumbnailImg.attr('src', videoData.thumbnail_url).show();
                    thumbnailPlaceholder.hide();
                    thumbnailStatus.html('✅ <span class="text-success">Available</span>');
                    
                    // Handle thumbnail load error
                    thumbnailImg.off('error').on('error', function() {
                        $(this).hide();
                        thumbnailPlaceholder.show();
                        thumbnailStatus.html('❌ <span class="text-warning">Failed to load</span>');
                    });
                } else {
                    $('#thumbnailPreview').hide();
                    $('#thumbnailPlaceholder').show();
                    $('#thumbnailStatus').html('❌ <span class="text-muted">Not available</span>');
                }
                
                // Store raw data for display
                window.currentRawData = videoData;
                
                // Show the result display
                $('#videoDetectionResult').show();
            }
            
            // Get platform-specific color for badges
            function getPlatformColor(platform) {
                const colors = {
                    'youtube': 'danger',
                    'vimeo': 'info', 
                    'facebook': 'primary',
                    'tiktok': 'dark',
                    'dailymotion': 'warning',
                    'wistia': 'success'
                };
                return colors[platform.toLowerCase()] || 'secondary';
            }

            // Enhanced preview embed button
            $('#previewEmbedBtn').click(function() {
                const embedCode = $('#embedCodeInput').val();
                
                if (embedCode) {
                    // Create modal for embed preview
                    const modalHtml = `
                        <div class="modal fade" id="embedPreviewModal" tabindex="-1">
                            <div class="modal-dialog modal-lg">
                                <div class="modal-content">
                                    <div class="modal-header">
                                        <h5 class="modal-title">
                                            <i class="fas fa-play me-2"></i>Video Embed Preview
                                        </h5>
                                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                                    </div>
                                    <div class="modal-body">
                                        <div class="embed-responsive">
                                            ${embedCode}
                                        </div>
                                    </div>
                                    <div class="modal-footer">
                                        <small class="text-muted me-auto">
                                            Platform: <strong>${$('#platformBadge').text()}</strong> | 
                                            Video ID: <code>${$('#extractedVideoId').text()}</code>
                                        </small>
                                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                                    </div>
                                </div>
                            </div>
                        </div>
                    `;
                    
                    // Remove existing modal and add new one
                    $('#embedPreviewModal').remove();
                    $('body').append(modalHtml);
                    
                    // Show modal
                    new bootstrap.Modal(document.getElementById('embedPreviewModal')).show();
                    
                } else {
                    showAlert('No embed code available to preview.', 'warning');
                }
            });
            
            // Raw data display toggle
            $(document).on('click', '#showRawDataBtn', function() {
                const rawDataDisplay = $('#rawDataDisplay');
                const rawDataContent = $('#rawDataContent');
                
                if (rawDataDisplay.is(':visible')) {
                    rawDataDisplay.hide();
                    $(this).html('<i class="fas fa-code me-1"></i>Raw Data');
                } else {
                    if (window.currentRawData) {
                        rawDataContent.text(JSON.stringify(window.currentRawData, null, 2));
                        rawDataDisplay.show();
                        $(this).html('<i class="fas fa-eye-slash me-1"></i>Hide Data');
                    } else {
                        showAlert('No raw data available.', 'warning');
                    }
                }
            });

            // Real-time auto-detection on URL input with instant feedback
            let urlDetectionTimeout;
            let currentDetectionRequest;
            
            // Enhanced URL input handler with visual debugging
            $('#videoUrl').on('input paste keyup', function(e) {
                clearTimeout(urlDetectionTimeout);
                
                // Cancel any ongoing request
                if (currentDetectionRequest) {
                    currentDetectionRequest.abort();
                    debugLog('🚫 Cancelled previous detection request', 'warning');
                }
                
                const url = $(this).val().trim();
                const detectBtn = $('#detectVideoBtn');
                const resultDiv = $('#videoDetectionResult');
                
                debugLog(`🎥 URL input event (${e.type}): "${url}"`, 'info');
                
                // Clear previous results immediately
                resultDiv.hide();
                window.currentRawData = null;
                
                // If URL is empty, reset state
                if (!url) {
                    detectBtn.removeClass('btn-outline-success btn-outline-danger').addClass('btn-outline-secondary');
                    detectBtn.html('<i class="fas fa-search me-1"></i>Detect');
                    debugLog('🔄 URL cleared - reset to default state', 'info');
                    return;
                }
                
                // Quick validation - check if it looks like a video URL
                const hasValidDomain = /(?:youtube\.com|youtu\.be|vimeo\.com|facebook\.com|tiktok\.com|dailymotion\.com|wistia\.com)/i.test(url);
                debugLog(`🔍 Domain validation: ${hasValidDomain ? '✅ Valid' : '❌ Invalid'}`, hasValidDomain ? 'success' : 'warning');
                
                if (url.length > 10 && hasValidDomain) {
                    // Show immediate feedback that URL looks valid
                    detectBtn.removeClass('btn-outline-secondary btn-outline-danger').addClass('btn-outline-success');
                    detectBtn.html('<i class="fas fa-check me-1"></i>Valid URL');
                    debugLog('✅ URL appears valid - starting auto-detection countdown', 'success');
                    
                    // Auto-detect after short delay
                    urlDetectionTimeout = setTimeout(function() {
                        debugLog('🚀 Auto-detection timeout triggered - starting detection...', 'info');
                        performAutoDetection(url);
                    }, 500); // Much faster - 0.5 seconds
                    
                } else if (url.length > 10) {
                    // URL doesn't look like a supported platform
                    detectBtn.removeClass('btn-outline-secondary btn-outline-success').addClass('btn-outline-danger');
                    detectBtn.html('<i class="fas fa-exclamation me-1"></i>Invalid URL');
                    debugLog('❌ URL domain not supported', 'error');
                } else {
                    // URL too short, show neutral state
                    detectBtn.removeClass('btn-outline-success btn-outline-danger').addClass('btn-outline-secondary');
                    detectBtn.html('<i class="fas fa-search me-1"></i>Detect');
                    debugLog('⚠️ URL too short for validation', 'warning');
                }
            });
            
            // Perform automatic detection
            function performAutoDetection(videoUrl) {
                const detectBtn = $('#detectVideoBtn');
                const resultDiv = $('#videoDetectionResult');
                
                console.log('🚀 Performing auto-detection for:', videoUrl);
                
                // Start sync process
                startSyncProcess();
                
                // Update button to show detecting state
                detectBtn.html('<i class="fas fa-spinner fa-spin me-1"></i>Detecting...');
                detectBtn.prop('disabled', true);
                
                // Make AJAX request
                currentDetectionRequest = $.ajax({
                    url: 'ajax/detect_video.php',
                    method: 'POST',
                    data: { video_url: videoUrl },
                    dataType: 'json',
                    timeout: 10000, // 10 second timeout
                    success: function(response) {
                        console.log('🎉 Auto-detection response:', response);
                        
                        if (response.success && response.data && response.data.is_valid) {
                            const videoData = response.data;
                            
                            // Populate the dynamic display
                            populateDetectionDisplay(videoData);
                            
                            // Store data for form submission
                            $('#detectedPlatformInput').val(videoData.platform);
                            $('#videoIdExternal').val(videoData.video_id);
                            $('#embedCodeInput').val(videoData.embed_code);
                            
                            // Auto-fill title if empty
                            if (!$('#videoTitle').val() && videoData.suggested_title) {
                                $('#videoTitle').val(videoData.suggested_title);
                            }
                            
                            // Complete sync successfully
                            completeSyncProcess(true);
                            
                            // Update button to success state
                            detectBtn.removeClass('btn-outline-secondary btn-outline-danger').addClass('btn-outline-success');
                            detectBtn.html('<i class="fas fa-check me-1"></i>Detected!');
                            
                            // Show success message
                            showAlert(`🎉 ${videoData.platform_name} video detected and synced automatically!`, 'success');
                            
                        } else {
                            // Detection failed
                            const errorMsg = response.message || response.error || 'Video detection failed';
                            console.warn('❌ Detection failed:', errorMsg);
                            
                            completeSyncProcess(false, errorMsg);
                            
                            detectBtn.removeClass('btn-outline-secondary btn-outline-success').addClass('btn-outline-danger');
                            detectBtn.html('<i class="fas fa-times me-1"></i>Failed');
                            
                            showAlert('Unable to detect video: ' + errorMsg, 'warning');
                        }
                    },
                    error: function(xhr, status, error) {
                        if (status !== 'abort') { // Don't show error for aborted requests
                            console.error('🚨 Auto-detection error:', error);
                            
                            completeSyncProcess(false, 'Network error: ' + error);
                            
                            detectBtn.removeClass('btn-outline-secondary btn-outline-success').addClass('btn-outline-danger');
                            detectBtn.html('<i class="fas fa-wifi me-1"></i>Error');
                            
                            showAlert('Detection error: ' + error, 'danger');
                        }
                    },
                    complete: function() {
                        // Re-enable button after delay
                        setTimeout(function() {
                            detectBtn.prop('disabled', false);
                        }, 1000);
                        
                        // Clear the current request
                        currentDetectionRequest = null;
                    }
                });
            }

            // Bootstrap form validation
            var forms = document.querySelectorAll('.needs-validation');
            
            // Loop over them and prevent submission
            Array.prototype.slice.call(forms).forEach(function(form) {
                form.addEventListener('submit', function(event) {
                    if (!form.checkValidity()) {
                        event.preventDefault();
                        event.stopPropagation();
                        
                        // Show error notification
                        Toastify({
                            text: "Please fix the errors in the form before submitting",
                            duration: 3000,
                            close: true,
                            gravity: "top",
                            position: "center",
                            backgroundColor: "#e74a3b",
                        }).showToast();
                    } else {
                        // Show loading state
                        $('#submitBtn').html('<i class="fas fa-spinner fa-spin"></i> Uploading...').attr('disabled', true);
                    }
                    
                    form.classList.add('was-validated');
                }, false);
            });
            
            // Reset form confirmation
            $('#resetBtn').click(function() {
                if (confirm('Are you sure you want to reset the form? All entered data will be lost.')) {
                    $('#videoUploadForm').removeClass('was-validated').trigger('reset');
                    $('#previewContainer').slideUp(300);
                    $('#progressContainer').slideUp(300);
                    $('#thumbnailPreview').slideUp(300);
                    $('#uploadProgress').css('width', '0%');
                    
                    // Reset upload status
                    $('#uploadStatusCard').html(`
                        <div class="d-flex align-items-center">
                            <span class="badge bg-secondary me-2 px-3 py-2">Idle</span>
                            <p class="text-muted mb-0 ms-2">No upload in progress</p>
                        </div>
                    `);
                    
                    // Notify user
                    Toastify({
                        text: "Form has been reset",
                        duration: 3000,
                        gravity: "top",
                        position: "center",
                        backgroundColor: "#858796",
                    }).showToast();
                }
            });
            
            // Video file validation and handling
            $('#videoFile').change(function() {
                const file = this.files[0];
                if (file) {
                    // Check file type
                    const validTypes = ['video/mp4', 'video/webm', 'video/quicktime', 'video/x-msvideo'];
                    if (!validTypes.includes(file.type)) {
                        Toastify({
                            text: "Invalid file type. Please select a video file (MP4, WebM, MOV, AVI).",
                            duration: 5000,
                            close: true,
                            gravity: "top", 
                            position: "center",
                            backgroundColor: "#e74a3b",
                        }).showToast();
                        
                        // Reset file input
                        $(this).val('');
                        return;
                    }
                    
                    // Update UI to show we're processing the file
                    $('#uploadStatus').removeClass('bg-info bg-danger').addClass('bg-warning').text('Processing...');
                    
                    // Update file info
                    $('#fileName').text(file.name);
                    $('#fileSize').text((file.size / (1024*1024)).toFixed(2) + ' MB');
                    $('#progressContainer').slideDown(300);
                    
                    // Create and show video preview
                    const videoURL = URL.createObjectURL(file);
                    $('#videoPreview').attr('src', videoURL).on('loadedmetadata', function() {
                        // Get video duration
                        const duration = Math.round(this.duration);
                        const minutes = Math.floor(duration / 60);
                        const seconds = duration % 60;
                        $('#videoDuration').text(`${minutes}m ${seconds}s`);
                        
                        // Update status
                        $('#uploadStatus').removeClass('bg-warning').addClass('bg-success').text('Ready');
                        
                        // Animate progress bar for visual feedback
                        $('#uploadProgress').css('width', '100%');
                        
                        // Show the video preview container with a smooth animation
                        $('#previewContainer').slideDown(300);
                        
                        // Update the upload status in the side panel
                        $('#uploadStatusCard').html(`
                            <div class="d-flex align-items-center">
                                <span class="badge bg-success me-2">Ready</span>
                                <p class="mb-0">Video ready to upload: ${file.name}</p>
                            </div>
                        `);
                    }).on('error', function() {
                        $('#uploadStatus').removeClass('bg-warning').addClass('bg-danger').text('Error');
                        Toastify({
                            text: "Error loading video preview. The file may be corrupted.",
                            duration: 5000,
                            close: true,
                            gravity: "top",
                            position: "center",
                            backgroundColor: "#e74a3b",
                        }).showToast();
                    });
                } else {
                    // No file selected (user canceled selection)
                    $('#progressContainer').slideUp(300);
                    $('#previewContainer').slideUp(300);
                }
            });
            
            // Handle drag and drop events for the upload area
            const uploadArea = document.getElementById('uploadArea');
            
            ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
                uploadArea.addEventListener(eventName, preventDefaults, false);
            });
            
            function preventDefaults(e) {
                e.preventDefault();
                e.stopPropagation();
            }
            
            ['dragenter', 'dragover'].forEach(eventName => {
                uploadArea.addEventListener(eventName, highlight, false);
            });
            
            ['dragleave', 'drop'].forEach(eventName => {
                uploadArea.addEventListener(eventName, unhighlight, false);
            });
            
            function highlight() {
                uploadArea.classList.add('highlight');
            }
            
            function unhighlight() {
                uploadArea.classList.remove('highlight');
            }
            
            uploadArea.addEventListener('drop', handleDrop, false);
            
            function handleDrop(e) {
                const dt = e.dataTransfer;
                const files = dt.files;
                
                if (files.length > 0) {
                    document.getElementById('videoFile').files = files;
                    $('#videoFile').trigger('change');
                }
            }
            
            // Connect Browse button to file input
            $('#browseBtn').on('click', function() {
                $('#videoFile').click();
            });
            
            // Thumbnail file validation
            $('#thumbnailFile').change(function() {
                const file = this.files[0];
                if (file) {
                    // Check file type
                    const validTypes = ['image/jpeg', 'image/png', 'image/jpg'];
                    if (!validTypes.includes(file.type)) {
                        Toastify({
                            text: "Invalid file type. Please select an image file (JPEG, PNG).",
                            duration: 5000,
                            close: true,
                            gravity: "top",
                            position: "center",
                            backgroundColor: "#e74a3b",
                        }).showToast();
                        
                        // Reset file input
                        $(this).val('');
                        return;
                    }
                    
                    // Show preview
                    const thumbnailURL = URL.createObjectURL(file);
                    $('#thumbnailImg').attr('src', thumbnailURL);
                    $('#thumbnailPreview').show();
                }
            });
            
            // Global AJAX error handling
            $(document).ajaxError(function(event, jqxhr, settings, thrownError) {
                console.error("AJAX error:", thrownError);
                Toastify({
                    text: "An error occurred. Please try again or contact support.",
                    duration: 5000,
                    close: true,
                    gravity: "top",
                    position: "center",
                    backgroundColor: "#e74a3b",
                }).showToast();
            });
        });
    </script>
</body>
</html>
